import { Head, Link, usePage } from '@inertiajs/react';
import { motion } from 'framer-motion';
import { useState, useEffect, useMemo } from 'react';
import { SharedData } from '@/types';

// Define the PageProps interface
interface PageProps {
  page: {
    title: string;
    content: string;
    meta_description?: string;
    updated_at: string;
  };
}

// Component to display global notifications
const GlobalNotification = () => {
  type FlashMessageType = {
    success?: string;
    error?: string;
    message?: string;
    errors?: {
      email?: string;
    };
  };

  const pageProps = usePage<SharedData & { flash?: FlashMessageType }>().props;
  const flash = useMemo(() => pageProps.flash || {} as FlashMessageType, [pageProps.flash]);
  const [isVisible, setIsVisible] = useState(true);

  // Hide notification after 5 seconds
  useEffect(() => {
    if (flash) {
      const hasMessage = flash?.success || flash?.error || flash?.message || (typeof flash === 'string');
      if (hasMessage) {
        const timer = setTimeout(() => setIsVisible(false), 5000);
        return () => clearTimeout(timer);
      }
    }
  }, [flash]);

  // If no flash message, return null
  if (!flash) return null;
  if (!isVisible) return null;

  // Check for any valid message format
  const message = flash?.success || flash?.error || flash?.message || (typeof flash === 'string' ? flash : null);
  if (!message) return null;

  const isSuccess = !!flash?.success || (flash?.message && !flash?.error);

  return (
    <div className="fixed top-4 right-4 z-50 max-w-md">
      <div className={`rounded-lg p-4 shadow-lg ${isSuccess ? 'bg-green-50 border border-green-200 text-green-800 dark:bg-green-900/20 dark:border-green-800 dark:text-green-300' : 'bg-red-50 border border-red-200 text-red-800 dark:bg-red-900/20 dark:border-red-800 dark:text-red-300'}`}>
        <div className="flex items-center">
          {isSuccess ? (
            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-2" viewBox="0 0 20 20" fill="currentColor">
              <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
            </svg>
          ) : (
            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-2" viewBox="0 0 20 20" fill="currentColor">
              <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
            </svg>
          )}
          <p>{message}</p>
          <button
            onClick={() => setIsVisible(false)}
            className="ml-auto text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
          >
            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
              <path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd" />
            </svg>
          </button>
        </div>
      </div>
    </div>
  );
};

export default function Page({ page }: PageProps) {
  const { auth } = usePage<SharedData>().props;

  return (
    <>
      <Head title={page.title}>
        <link rel="preconnect" href="https://fonts.bunny.net" />
        <link href="https://fonts.bunny.net/css?family=instrument-sans:400,500,600|plus-jakarta-sans:500,600,700,800" rel="stylesheet" />
        <meta name="description" content={page.meta_description || ''} />
      </Head>

      {/* Global notification for redirects */}
      <GlobalNotification />

      <div className="min-h-screen bg-gradient-to-b from-white to-gray-50 dark:from-gray-950 dark:to-gray-900 text-gray-900 dark:text-gray-100">
        {/* Header */}
        <header className="relative w-full border-b border-gray-200 dark:border-gray-800 bg-white/80 dark:bg-gray-950/80 backdrop-blur-md">
          <div className="container mx-auto flex h-16 items-center justify-between px-4 lg:px-8">
            <motion.div
              initial={{ opacity: 0, y: -10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
              className="flex items-center gap-2"
            >
              <div className="h-8 w-8 rounded-lg bg-indigo-600 flex items-center justify-center text-white font-bold">SL</div>
              <span className="text-xl font-bold tracking-tight font-jakarta">SaaSLarkit</span>
            </motion.div>

            <nav className="flex items-center gap-6">
              {auth.user ? (
                <Link
                  href={route('dashboard')}
                  className="inline-flex h-9 items-center justify-center rounded-md bg-indigo-600 px-4 text-sm font-medium text-white shadow-sm transition-colors hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-950"
                >
                  Dashboard
                </Link>
              ) : (
                <>
                  <div className="hidden lg:flex items-center gap-6">
                    <Link href={route('home') + '#features'} className="text-sm font-medium hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">
                      Features
                    </Link>
                    <Link href={route('home') + '#pricing'} className="text-sm font-medium hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">
                      Pricing
                    </Link>
                    <Link href={route('home') + '#testimonials'} className="text-sm font-medium hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">
                      Testimonials
                    </Link>
                  </div>
                  <div className="flex items-center gap-3">
                    <Link
                      href={route('login')}
                      className="inline-flex h-9 items-center justify-center rounded-md border border-gray-200 bg-white px-4 text-sm font-medium shadow-sm transition-colors hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:border-gray-800 dark:bg-gray-900 dark:hover:bg-gray-800 dark:focus:ring-offset-gray-950"
                    >
                      Log in
                    </Link>
                    <Link
                      href={route('register')}
                      className="inline-flex h-9 items-center justify-center rounded-md bg-indigo-600 px-4 text-sm font-medium text-white shadow-sm transition-colors hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-950"
                    >
                      Sign up
                    </Link>
                  </div>
                </>
              )}
            </nav>
          </div>
        </header>

        {/* Main Content */}
        <div className="container mx-auto px-4 py-12 md:py-16 max-w-4xl">
          <h1 className="text-3xl font-bold tracking-tight text-gray-900 dark:text-white sm:text-4xl mb-8">
            {page.title}
          </h1>

          <div
            className="prose prose-indigo max-w-none dark:prose-invert"
            dangerouslySetInnerHTML={{ __html: page.content }}
          />

          <div className="mt-8 text-sm text-gray-500 dark:text-gray-400">
            Last updated: {page.updated_at}
          </div>
        </div>

        {/* Footer */}
        <footer className="border-t border-gray-200 bg-white py-12 dark:border-gray-800 dark:bg-gray-950">
          <div className="container px-4 md:px-6 mx-auto max-w-7xl">
            <div className="grid grid-cols-1 gap-8 md:grid-cols-2 lg:grid-cols-4">
              <div className="space-y-4">
                <div className="flex items-center gap-2">
                  <div className="h-8 w-8 rounded-lg bg-indigo-600 flex items-center justify-center text-white font-bold">SL</div>
                  <span className="text-xl font-bold tracking-tight font-jakarta">SaaSLarkit</span>
                </div>
                <p className="text-sm text-gray-500 dark:text-gray-400">
                  The complete SaaS starter kit for your next big idea.
                </p>
              </div>
              <div className="space-y-4">
                <h3 className="text-sm font-semibold">Product</h3>
                <ul className="space-y-2">
                  <li>
                    <Link href={route('home') + '#features'} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Features
                    </Link>
                  </li>
                  <li>
                    <Link href={route('home') + '#pricing'} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Pricing
                    </Link>
                  </li>
                  <li>
                    <Link href={route('home') + '#testimonials'} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Testimonials
                    </Link>
                  </li>
                </ul>
              </div>
              <div className="space-y-4">
                <h3 className="text-sm font-semibold">Company</h3>
                <ul className="space-y-2">
                  <li>
                    <Link href={route('home')} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      About
                    </Link>
                  </li>
                  <li>
                    <Link href={route('home')} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Blog
                    </Link>
                  </li>
                  <li>
                    <Link href={route('home')} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Careers
                    </Link>
                  </li>
                </ul>
              </div>
              <div className="space-y-4">
                <h3 className="text-sm font-semibold">Legal</h3>
                <ul className="space-y-2">
                  <li>
                    <Link href={route('pages.show', 'privacy-policy')} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Privacy Policy
                    </Link>
                  </li>
                  <li>
                    <Link href={route('pages.show', 'terms-of-service')} className="text-sm text-gray-500 hover:text-gray-900 dark:text-gray-400 dark:hover:text-gray-50">
                      Terms of Service
                    </Link>
                  </li>
                </ul>
              </div>
            </div>
            <div className="mt-8 border-t border-gray-200 pt-8 dark:border-gray-800">
              <p className="text-sm text-gray-500 dark:text-gray-400">
                © {new Date().getFullYear()} SaaSLarkit. All rights reserved.
              </p>
            </div>
          </div>
        </footer>
      </div>
    </>
  );
}
