import { Link } from '@inertiajs/react';
import { motion } from 'framer-motion';
import { RefObject } from 'react';

type WhatToExpectType = {
    id: number;
    title: string;
    description: string;
    icon: string;
    order: number;
    is_active: boolean;
    created_at: string;
    updated_at: string;
};

interface ProjectInfoSectionProps {
    whatToExpect: WhatToExpectType[];
    emailSectionRef: RefObject<HTMLDivElement>;
}

const ProjectInfoSection = ({ whatToExpect, emailSectionRef }: ProjectInfoSectionProps) => {
    return (
        <section className="py-16 md:py-20 bg-gradient-to-b from-white to-indigo-50 dark:from-gray-950 dark:to-indigo-950/20">
            <div className="container px-4 md:px-6 mx-auto max-w-7xl">
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.5 }}
                    className="text-center mb-12 md:mb-16"
                >
                    <div className="inline-block rounded-full bg-indigo-100 px-3 py-1 text-sm font-medium text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300">
                        Coming Q3 2025
                    </div>
                    <h2 className="mt-4 text-3xl font-bold tracking-tight sm:text-4xl md:text-5xl font-jakarta">Revolutionizing SaaS Development</h2>
                    <p className="mt-4 text-gray-500 md:text-xl/relaxed lg:text-base/relaxed xl:text-xl/relaxed dark:text-gray-400 max-w-3xl mx-auto">
                        We're building the next generation platform that will change how developers create and scale SaaS applications.
                    </p>
                </motion.div>

                <div className="grid grid-cols-1 gap-8 lg:grid-cols-2 lg:gap-16">
                    <motion.div
                        initial={{ opacity: 0, x: -20 }}
                        whileInView={{ opacity: 1, x: 0 }}
                        viewport={{ once: true }}
                        transition={{ duration: 0.6 }}
                        className="space-y-6"
                    >
                        <h3 className="text-2xl font-bold tracking-tight font-jakarta">Our Vision</h3>
                        <p className="text-gray-600 dark:text-gray-400">
                            We believe that building a successful SaaS product shouldn't require months of development time and extensive resources. Our platform is designed to democratize SaaS development, making it accessible to solo founders, small teams, and established companies alike.
                        </p>

                        <div className="space-y-4">
                            {[
                                {
                                    title: "Launch Faster",
                                    description: "Reduce your time-to-market from months to days with our pre-built components and infrastructure."
                                },
                                {
                                    title: "Scale Confidently",
                                    description: "Built with scalability in mind from day one, so your application can grow with your user base."
                                },
                                {
                                    title: "Focus on Your Core Product",
                                    description: "Stop reinventing the wheel with auth, billing, and user management. Focus on what makes your product unique."
                                }
                            ].map((item, index) => (
                                <div key={index} className="flex gap-3">
                                    <div className="flex-shrink-0 h-6 w-6 rounded-full bg-indigo-100 flex items-center justify-center text-indigo-600 dark:bg-indigo-900/30 dark:text-indigo-400">
                                        <svg xmlns="http://www.w3.org/2000/svg" className="h-4 w-4" viewBox="0 0 20 20" fill="currentColor">
                                            <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                                        </svg>
                                    </div>
                                    <div>
                                        <h4 className="font-semibold">{item.title}</h4>
                                        <p className="text-sm text-gray-500 dark:text-gray-400">{item.description}</p>
                                    </div>
                                </div>
                            ))}
                        </div>
                    </motion.div>

                    <motion.div
                        initial={{ opacity: 0, x: 20 }}
                        whileInView={{ opacity: 1, x: 0 }}
                        viewport={{ once: true }}
                        transition={{ duration: 0.6, delay: 0.2 }}
                        className="space-y-6"
                    >
                        <h3 className="text-2xl font-bold tracking-tight font-jakarta">What to Expect</h3>
                        <p className="text-gray-600 dark:text-gray-400">
                            Our platform will provide everything you need to build, launch, and scale your SaaS product. Here's what you can look forward to:
                        </p>

                        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 lg:grid-cols-3">
                            {whatToExpect.length > 0 ? (
                                whatToExpect.map((item, index) => (
                                    <motion.div
                                        key={item.id}
                                        initial={{ opacity: 0, y: 20 }}
                                        whileInView={{ opacity: 1, y: 0 }}
                                        viewport={{ once: true }}
                                        transition={{ duration: 0.5, delay: index * 0.1 }}
                                        className="flex flex-col items-center p-4 text-center rounded-xl border border-gray-200 bg-white shadow-sm transition-all hover:shadow-md dark:border-gray-800 dark:bg-gray-950"
                                    >
                                        <div
                                            className="h-12 w-12 flex items-center justify-center rounded-full bg-indigo-100 text-indigo-600 mb-4 dark:bg-indigo-900/30 dark:text-indigo-400 *:w-6 *:h-6"
                                            dangerouslySetInnerHTML={{ __html: item.icon }}
                                        />
                                        <h4 className="text-lg font-semibold">{item.title}</h4>
                                        <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">{item.description}</p>
                                    </motion.div>
                                ))
                            ) : (
                                // Fallback static content if no items in database
                                [
                                    {
                                        icon: (
                                            <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" viewBox="0 0 20 20" fill="currentColor">
                                                <path d="M7 3a1 1 0 000 2h6a1 1 0 100-2H7zM4 7a1 1 0 011-1h10a1 1 0 110 2H5a1 1 0 01-1-1zM2 11a2 2 0 012-2h12a2 2 0 012 2v4a2 2 0 01-2 2H4a2 2 0 01-2-2v-4z" />
                                            </svg>
                                        ),
                                        title: "Customizable Templates",
                                        description: "Start with a template that fits your needs and customize it to match your brand."
                                    },
                                    {
                                        icon: (
                                            <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" viewBox="0 0 20 20" fill="currentColor">
                                                <path fillRule="evenodd" d="M12.316 3.051a1 1 0 01.633 1.265l-4 12a1 1 0 11-1.898-.632l4-12a1 1 0 011.265-.633zM5.707 6.293a1 1 0 010 1.414L3.414 10l2.293 2.293a1 1 0 11-1.414 1.414l-3-3a1 1 0 010-1.414l3-3a1 1 0 011.414 0zm8.586 0a1 1 0 011.414 0l3 3a1 1 0 010 1.414l-3 3a1 1 0 11-1.414-1.414L16.586 10l-2.293-2.293a1 1 0 010-1.414z" clipRule="evenodd" />
                                            </svg>
                                        ),
                                        title: "Developer API",
                                        description: "Extend functionality with our comprehensive API and webhooks."
                                    },
                                    {
                                        icon: (
                                            <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" viewBox="0 0 20 20" fill="currentColor">
                                                <path d="M8.433 7.418c.155-.103.346-.196.567-.267v1.698a2.305 2.305 0 01-.567-.267C8.07 8.34 8 8.114 8 8c0-.114.07-.34.433-.582zM11 12.849v-1.698c.22.071.412.164.567.267.364.243.433.468.433.582 0 .114-.07.34-.433.582a2.305 2.305 0 01-.567.267z" />
                                                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-13a1 1 0 10-2 0v.092a4.535 4.535 0 00-1.676.662C6.602 6.234 6 7.009 6 8c0 .99.602 1.765 1.324 2.246.48.32 1.054.545 1.676.662v1.941c-.391-.127-.68-.317-.843-.504a1 1 0 10-1.51 1.31c.562.649 1.413 1.076 2.353 1.253V15a1 1 0 102 0v-.092a4.535 4.535 0 001.676-.662C13.398 13.766 14 12.991 14 12c0-.99-.602-1.765-1.324-2.246A4.535 4.535 0 0011 9.092V7.151c.391.127.68.317.843.504a1 1 0 101.511-1.31c-.563-.649-1.413-1.076-2.354-1.253V5z" clipRule="evenodd" />
                                            </svg>
                                        ),
                                        title: "Subscription Management",
                                        description: "Built-in billing and subscription tools with multiple payment providers."
                                    },
                                    {
                                        icon: (
                                            <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" viewBox="0 0 20 20" fill="currentColor">
                                                <path fillRule="evenodd" d="M2.166 4.999A11.954 11.954 0 0010 1.944A11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                                            </svg>
                                        ),
                                        title: "Enterprise Security",
                                        description: "Bank-level security with data encryption, 2FA, and compliance features."
                                    },
                                    {
                                        icon: (
                                            <svg xmlns="http://www.w3.org/2000/svg" className="h-6 w-6" viewBox="0 0 20 20" fill="currentColor">
                                                <path d="M5 3a2 2 0 00-2 2v2a2 2 0 002 2h2a2 2 0 002-2V5a2 2 0 00-2-2H5zM5 11a2 2 0 00-2 2v2a2 2 0 002 2h2a2 2 0 002-2v-2a2 2 0 00-2-2H5zM11 5a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2V5zM11 13a2 2 0 012-2h2a2 2 0 012 2v2a2 2 0 01-2 2h-2a2 2 0 01-2-2v-2z" />
                                            </svg>
                                        ),
                                        title: "AI-Powered Insights",
                                        description: "Leverage machine learning to gain valuable insights about your users and optimize your business."
                                    }
                                ].map((item, index) => (
                                    <motion.div
                                        key={index}
                                        initial={{ opacity: 0, y: 20 }}
                                        whileInView={{ opacity: 1, y: 0 }}
                                        viewport={{ once: true }}
                                        transition={{ duration: 0.5, delay: index * 0.1 }}
                                        className="flex flex-col items-center p-4 text-center rounded-xl border border-gray-200 bg-white shadow-sm transition-all hover:shadow-md dark:border-gray-800 dark:bg-gray-950"
                                    >
                                        <div className="h-12 w-12 flex items-center justify-center rounded-full bg-indigo-100 text-indigo-600 mb-4 dark:bg-indigo-900/30 dark:text-indigo-400">
                                            {item.icon}
                                        </div>
                                        <h4 className="text-lg font-semibold">{item.title}</h4>
                                        <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">{item.description}</p>
                                    </motion.div>
                                ))
                            )}
                        </div>
                    </motion.div>
                </div>

                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.5, delay: 0.4 }}
                    className="mt-12 md:mt-16 flex flex-col items-center text-center"
                >
                    <div className="max-w-2xl">
                        <h3 className="text-xl font-bold mb-4">Be the First to Know When We Launch</h3>
                        <p className="text-gray-600 dark:text-gray-400 mb-6">
                            Join our waitlist to receive exclusive early access, special pricing, and updates on our development progress.
                        </p>
                        <Link
                            href="#email-signup"
                            className="inline-flex h-10 items-center justify-center rounded-md bg-indigo-600 px-8 text-sm font-medium text-white shadow-sm transition-colors hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-950"
                            onClick={(e) => {
                                e.preventDefault();
                                emailSectionRef.current?.scrollIntoView({ behavior: 'smooth' });
                            }}
                        >
                            Join the Waitlist
                        </Link>
                    </div>
                </motion.div>
            </div>
        </section>
    );
};

export default ProjectInfoSection;
