import { motion } from 'framer-motion';

type FaqType = {
    id: number;
    question: string;
    answer: string;
    order: number;
    is_active: boolean;
    created_at: string;
    updated_at: string;
};

interface FaqSectionProps {
    faqs: FaqType[];
}

const FaqSection = ({ faqs }: FaqSectionProps) => {
    return (
        <section className="py-16 md:py-20 bg-gray-50 dark:bg-gray-900">
            <div className="container px-4 md:px-6 mx-auto max-w-7xl">
                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.5 }}
                    className="text-center mb-12 md:mb-16"
                >
                    <h2 className="text-3xl font-bold tracking-tight sm:text-4xl md:text-5xl font-jakarta">Frequently Asked Questions</h2>
                    <p className="mt-4 text-gray-500 md:text-xl/relaxed lg:text-base/relaxed xl:text-xl/relaxed dark:text-gray-400 max-w-3xl mx-auto">
                        Everything you need to know about our upcoming SaaS platform.
                    </p>
                </motion.div>

                <div className="grid gap-6 md:gap-8 max-w-3xl mx-auto">
                    {faqs.length > 0 ? (
                        faqs.map((faq: FaqType, index: number) => (
                            <motion.div
                                key={faq.id}
                                initial={{ opacity: 0, y: 10 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ duration: 0.4, delay: index * 0.1 }}
                                className="rounded-lg border border-gray-200 bg-white p-6 shadow-sm dark:border-gray-800 dark:bg-gray-950"
                            >
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{faq.question}</h3>
                                <div
                                    className="mt-2 text-gray-600 dark:text-gray-400 prose prose-sm dark:prose-invert max-w-none"
                                    dangerouslySetInnerHTML={{ __html: faq.answer }}
                                />
                            </motion.div>
                        ))
                    ) : (
                        // Fallback content if no FAQs are available
                        [
                            {
                                question: "When will the platform be available?",
                                answer: "We're targeting a launch in Q3 2025. Join our waitlist to be notified as soon as we launch and to get early access."
                            },
                            {
                                question: "How much will it cost?",
                                answer: "We'll offer multiple pricing tiers to accommodate different needs and team sizes. Early adopters from our waitlist will receive special discounted pricing."
                            },
                            {
                                question: "Will there be a free trial?",
                                answer: "Yes, we plan to offer a 14-day free trial so you can explore all features before committing to a subscription."
                            }
                        ].map((item, index) => (
                            <motion.div
                                key={index}
                                initial={{ opacity: 0, y: 10 }}
                                whileInView={{ opacity: 1, y: 0 }}
                                viewport={{ once: true }}
                                transition={{ duration: 0.4, delay: index * 0.1 }}
                                className="rounded-lg border border-gray-200 bg-white p-6 shadow-sm dark:border-gray-800 dark:bg-gray-950"
                            >
                                <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{item.question}</h3>
                                <p className="mt-2 text-gray-600 dark:text-gray-400">{item.answer}</p>
                            </motion.div>
                        ))
                    )}
                </div>

                <motion.div
                    initial={{ opacity: 0, y: 20 }}
                    whileInView={{ opacity: 1, y: 0 }}
                    viewport={{ once: true }}
                    transition={{ duration: 0.5, delay: 0.6 }}
                    className="mt-12 text-center"
                >
                    <p className="text-gray-600 dark:text-gray-400 mb-6">Still have questions?</p>
                    <a
                        href="mailto:info@saaslarkit.com"
                        className="inline-flex h-10 items-center justify-center rounded-md bg-indigo-600 px-8 text-sm font-medium text-white shadow-sm transition-colors hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-950"
                    >
                        Contact Us
                    </a>
                </motion.div>
            </div>
        </section>
    );
};

export default FaqSection;
