import React from 'react';
import { motion } from 'framer-motion';
import FeatureItem from './FeatureItem';
import { Button } from '@/components/ui/button';

type Plan = {
  id: number;
  name: string;
  slug: string;
  billing_interval: string;
  price_cents: number;
  description?: string;
  features: {
    feature: string;
    enabled: boolean;
  }[];
  is_active: boolean;
  formattedPrice: string;
  formattedInterval: string;
  lemonsqueezy_price_id: string;
  lemonsqueezy_product_id: string;
};

type Subscription = {
  id: number;
  billable_type: string;
  billable_id: number;
  type: string;
  lemon_squeezy_id: string;
  status: string;
  product_id: string;
  variant_id: string;
  card_brand?: string;
  card_last_four?: string;
  pause_mode: string | null;
  pause_resumes_at: string | null;
  trial_ends_at: string | null;
  renews_at?: string;
  ends_at?: string;
  created_at: string;
  updated_at: string;
  active: boolean;
  cancelled: boolean;
  ended: () => boolean;
};

type PlanCardProps = {
  plan: Plan;
  billingInterval: 'monthly' | 'yearly';
  activeSubscription: Subscription | null;
  onChangePlan: (planId: number) => void;
};

const PlanCard: React.FC<PlanCardProps> = ({
  plan,
  billingInterval,
  activeSubscription,
  onChangePlan
}) => {

  const features = plan.features;
  const isCurrentPlan = activeSubscription && activeSubscription.product_id === plan.lemonsqueezy_product_id;
  const isPremiumPlan = plan.name.toLowerCase().includes('pro');

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.5 }}
      className={`relative rounded-2xl border ${isCurrentPlan ? 'border-indigo-500 ring-2 ring-indigo-200' : 'border-gray-200'} bg-white p-6 shadow-sm dark:border-gray-800 dark:bg-gray-900 flex flex-col h-full`}
    >
      {isPremiumPlan && (
        <div className="absolute top-0 right-0 -translate-y-1/2 translate-x-0 rounded-full bg-green-100 px-3 py-1 text-xs font-medium text-green-800 dark:bg-green-900/30 dark:text-green-300">
          Best Value
        </div>
      )}
      {isCurrentPlan && (
        <div className="absolute top-0 left-0 -translate-y-1/2 translate-x-0 rounded-full bg-indigo-100 px-3 py-1 text-xs font-medium text-indigo-800 dark:bg-indigo-900/30 dark:text-indigo-300">
          Current Plan
        </div>
      )}
      <div className="mb-5">
        <h3 className="text-lg font-semibold">{plan.name}</h3>
        <div className="mt-2 flex items-baseline">
          <span className="text-3xl font-bold">${(plan.price_cents / 100).toFixed(2)}</span>
          <span className="text-gray-500 dark:text-gray-400 ml-1">/{plan.billing_interval === 'monthly' ? 'month' : 'year'}</span>
        </div>
        <p className="mt-3 text-sm text-gray-500 dark:text-gray-400">{plan.description}</p>
        {billingInterval === 'yearly' && (
          <p className="mt-1 text-xs text-green-600 dark:text-green-400">
            Includes 4 months free
          </p>
        )}
      </div>

      <div className="mt-2 space-y-3 flex-grow">
        <h4 className="text-sm font-medium">What's included:</h4>
        <ul className="space-y-3">
          {Array.isArray(features) && features.length > 0 ? (
            features.map((feature, index) => (
              <FeatureItem
                key={index}
                feature={feature.feature}
                enabled={feature.enabled}
              />
            ))
          ) : (
            <li className="text-sm text-gray-500">No features available</li>
          )}
        </ul>
      </div>

      <div className="mt-6">
        {isCurrentPlan ? (
          <Button className="w-full" disabled>Current Plan</Button>
        ) : activeSubscription ? (
          <Button
            className="w-full"
            onClick={() => onChangePlan(plan.id)}
            disabled={activeSubscription.cancelled}
            variant="outline"
          >
            Change Plan
          </Button>
        ) : (
          <a
            href={route('subscription.checkout', plan.id)}
            className="inline-flex w-full items-center justify-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white shadow-sm transition-colors hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:ring-offset-2 dark:focus:ring-offset-gray-950"
          >
            Subscribe
          </a>
        )}
      </div>
    </motion.div>
  );
};

export default PlanCard;
