import React, { useState } from 'react';
import { usePlans } from '@/hooks/usePlans';
import BillingToggle from './BillingToggle';
import PlanCard from './PlanCard';

interface PricingPlansProps {
    auth: { user: any /* eslint-disable-line @typescript-eslint/no-explicit-any */ } | null;
}

const PricingPlans: React.FC<PricingPlansProps> = ({ auth }) => {
    const { plans, loading, error } = usePlans();
    const [billingInterval, setBillingInterval] = useState<'monthly' | 'yearly'>('monthly');

    // Loading state
    if (loading) {
        return (
            <div className="flex justify-center items-center py-12">
                <div className="h-8 w-8 animate-spin rounded-full border-4 border-indigo-500 border-t-transparent"></div>
            </div>
        );
    }

    // Error state
    if (error) {
        return (
            <div className="text-center py-8">
                <p className="text-red-500 dark:text-red-400">{error}</p>
            </div>
        );
    }

    // Empty state
    if (plans.length === 0) {
        return (
            <div className="text-center py-8">
                <p className="text-gray-500 dark:text-gray-400">No pricing plans are currently available.</p>
            </div>
        );
    }

    // Filter plans by billing interval
    const filteredPlans = plans.filter(plan => plan.billing_interval === billingInterval);

    const yearlySavingsPercentage = 33.33; // Approximately 4 months free (33.33%)

    return (
        <div className="flex flex-col items-center">
            <BillingToggle
                billingInterval={billingInterval}
                setBillingInterval={setBillingInterval}
                yearlySavingsPercentage={yearlySavingsPercentage}
            />

            <div className="grid grid-cols-1 gap-8 md:grid-cols-2 lg:max-w-5xl lg:mx-auto w-full">
                {filteredPlans.length > 0 ? (
                    filteredPlans.map((plan) => (
                        <PlanCard
                            key={plan.id}
                            plan={plan}
                            billingInterval={billingInterval}
                            auth={auth}
                        />
                    ))
                ) : (
                    <div className="col-span-2 text-center py-8">
                        <p className="text-gray-500 dark:text-gray-400">
                            No {billingInterval} pricing plans are currently available.
                        </p>
                    </div>
                )}
            </div>
        </div>
    );
};

export default PricingPlans;
