<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use LemonSqueezy\Laravel\Subscription;

class SubscriptionController extends Controller
{
    /**
     * Show the subscription management page
     */
    public function index()
    {
        $user = Auth::user();
        $subscriptions = [];
        $activeSubscription = null;

        if ($user) {
            // Get all of the user's subscriptions
            $subscriptions = $user->subscriptions;

            // Find the active subscription if any
            $activeSubscription = $subscriptions->first(function ($subscription) {
                return $subscription->active();
            });
        }

        // Get all available plans
        $plans = Plan::where('is_active', true)->orderBy('price_cents', )->get();

        return Inertia::render('subscription/manage', [
            'subscriptions' => $subscriptions,
            'activeSubscription' => $activeSubscription,
            'plans' => $plans
        ]);
    }

    /**
     * Create a checkout for the given plan
     */
    public function checkout($planId)
    {
        $user = Auth::user();
        $plan = Plan::findOrFail($planId);

        if (!$plan->is_active) {
            return redirect()->back()->with('error', 'This plan is not available.');
        }

        // Add this before the checkout creation in SubscriptionController.php
        \Log::info('Creating checkout with price ID: ' . $plan->lemonsqueezy_price_id);
        // Create a checkout for the plan using Lemon Squeezy
        $checkout = $user->checkout($plan->lemonsqueezy_price_id, [
            'success_url' => route('subscription.success'),
            'cancel_url' => route('subscription.index'),
            'checkout' => [
                'name' => $user->name,
                'email' => $user->email,
                'custom' => [
                    'plan_id' => $plan->id,
                ],
            ],
        ]);

        // Get the URL using the url() method of the Checkout object
        return redirect($checkout->url());
    }

    /**
     * Handle successful subscription
     */
    public function success()
    {
        return redirect()->route('subscription.index')->with('success', 'Your subscription was successful!');
    }

    /**
     * Cancel the user's subscription
     */
    public function cancel(Request $request)
    {
        $user = Auth::user();
        $subscriptionId = $request->subscription_id;

        $subscription = $user->subscriptions()->where('id', $subscriptionId)->first();

        if ($subscription && $subscription->active()) {
            // This will cancel the subscription at the end of the current billing period
            $subscription->cancel();

            return redirect()->route('subscription.index')
                ->with('success', 'Your subscription has been cancelled and will end at the end of the billing period.');
        }

        return redirect()->route('subscription.index')
            ->with('error', 'Subscription could not be found or is already cancelled.');
    }

    /**
     * Change the subscription plan
     */
    public function changePlan(Request $request)
    {
        $user = Auth::user();
        $subscriptionId = $request->subscription_id;
        $newPlanId = $request->plan_id;

        $subscription = $user->subscriptions()->where('id', $subscriptionId)->first();
        $newPlan = Plan::findOrFail($newPlanId);

        if ($subscription && $subscription->active()) {
            // Change the subscription to the new plan
            $subscription->swap($newPlan->lemonsqueezy_product_id, $newPlan->lemonsqueezy_price_id);

            return redirect()->route('subscription.index')
                ->with('success', 'Your subscription plan has been updated.');
        }

        return redirect()->route('subscription.index')
            ->with('error', 'Subscription could not be found or is inactive.');
    }

    /**
     * Resume a cancelled subscription if it's still within the billing period
     */
    public function resume(Request $request)
    {
        $user = Auth::user();
        $subscriptionId = $request->subscription_id;

        $subscription = $user->subscriptions()->where('id', $subscriptionId)->first();

        if ($subscription && $subscription->cancelled() && !$subscription->ended()) {
            // Resume the subscription
            $subscription->resume();

            return redirect()->route('subscription.index')
                ->with('success', 'Your subscription has been resumed.');
        }

        return redirect()->route('subscription.index')
            ->with('error', 'Subscription could not be resumed.');
    }
}
