<?php

namespace App\Filament\Widgets;

use App\Models\User;
use App\Models\Plan;
use App\Models\LemonSqueezySubscription;
use Illuminate\Support\Facades\DB;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Carbon\Carbon;

class SubscriptionStatsWidget extends BaseWidget
{
    protected function getStats(): array
    {
        // Total active subscribers from lemon_squeezy_subscriptions
        $totalSubscribers = LemonSqueezySubscription::where('status', 'active')
            ->distinct('billable_id')
            ->count('billable_id');
        
        // New subscribers this month
        $newSubscribersThisMonth = LemonSqueezySubscription::where('status', 'active')
            ->whereMonth('created_at', Carbon::now()->month)
            ->distinct('billable_id')
            ->count('billable_id');
        
        // Active plans count
        $activePlans = Plan::where('is_active', true)->count();
        
        // Calculate MRR (Monthly Recurring Revenue) in dollars
        $mrr = 0;
        
        // Get all active subscriptions
        $activeSubscriptions = LemonSqueezySubscription::where('status', 'active')->get();
        
        foreach ($activeSubscriptions as $subscription) {
            // Find the plan for this subscription using the variant_id which corresponds to lemonsqueezy_price_id
            $plan = Plan::where('lemonsqueezy_price_id', $subscription->variant_id)->first();
            
            if ($plan) {
                if ($plan->billing_interval === 'monthly') {
                    $mrr += $plan->price_cents / 100;
                } else if ($plan->billing_interval === 'yearly') {
                    // Convert yearly price to monthly equivalent
                    $mrr += ($plan->price_cents / 100) / 12;
                }
            }
        }
        
        return [
            Stat::make('Total Subscribers', $totalSubscribers)
                ->description('Active paying customers')
                ->descriptionIcon('heroicon-m-users')
                ->color('success'),
                
            Stat::make('New Subscribers This Month', $newSubscribersThisMonth)
                ->description('Since ' . Carbon::now()->startOfMonth()->format('M d, Y'))
                ->descriptionIcon('heroicon-m-user-plus')
                ->color('info'),
                
            Stat::make('Monthly Recurring Revenue', '$' . number_format($mrr, 2))
                ->description('Estimated MRR')
                ->descriptionIcon('heroicon-m-currency-dollar')
                ->color('warning'),
        ];
    }
}
