<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Filament\Resources\UserResource\Pages\ListUsers;
use App\Filament\Resources\UserResource\Pages\CreateUser;
use App\Filament\Resources\UserResource\Pages\EditUser;
use App\Filament\Resources\UserResource\RelationManagers;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';
    
    protected static ?string $navigationGroup = 'User Management';

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('User Information')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->required(),
                        Forms\Components\TextInput::make('email')
                            ->email()
                            ->required(),
                        Forms\Components\DateTimePicker::make('email_verified_at'),
                        Forms\Components\TextInput::make('password')
                            ->password()
                            ->dehydrated(fn ($state) => filled($state))
                            ->dehydrateStateUsing(fn ($state) => bcrypt($state))
                            ->required(fn (string $operation): bool => $operation === 'create'),
                    ])->columns(2),
                Forms\Components\Section::make('Subscription Information')
                    ->schema([
                        Forms\Components\Placeholder::make('subscription_status')
                            ->label('Subscription Status')
                            ->content(function ($record) {
                                if (!$record) return 'N/A'; // For create form
                                
                                $activeSubscription = $record->lemonSqueezySubscriptions()
                                    ->where('status', 'active')
                                    ->first();
                                    
                                return $activeSubscription ? 'Active' : 'Not Subscribed';
                            }),
                        Forms\Components\Placeholder::make('subscription_id')
                            ->label('Subscription ID')
                            ->content(function ($record) {
                                if (!$record) return 'N/A'; // For create form
                                
                                $activeSubscription = $record->lemonSqueezySubscriptions()
                                    ->where('status', 'active')
                                    ->first();
                                    
                                return $activeSubscription ? $activeSubscription->id : 'N/A';
                            }),
                        Forms\Components\Placeholder::make('plan_name')
                            ->label('Plan Name')
                            ->content(function ($record) {
                                if (!$record) return 'N/A'; // For create form
                                
                                $activeSubscription = $record->lemonSqueezySubscriptions()
                                    ->where('status', 'active')
                                    ->first();
                                    
                                return $activeSubscription && $activeSubscription->plan ? $activeSubscription->plan->name : 'N/A';
                            }),
                        Forms\Components\Placeholder::make('ends_at')
                            ->label('Subscription Ends At')
                            ->content(function ($record) {
                                if (!$record) return 'N/A'; // For create form
                                
                                $activeSubscription = $record->lemonSqueezySubscriptions()
                                    ->where('status', 'active')
                                    ->first();
                                    
                                return $activeSubscription && $activeSubscription->ends_at ? $activeSubscription->ends_at->format('Y-m-d H:i:s') : 'N/A';
                            }),
                        Forms\Components\Placeholder::make('trial_ends_at')
                            ->label('Trial Ends At')
                            ->content(function ($record) {
                                if (!$record) return 'N/A'; // For create form
                                
                                $activeSubscription = $record->lemonSqueezySubscriptions()
                                    ->where('status', 'active')
                                    ->first();
                                    
                                return $activeSubscription && $activeSubscription->trial_ends_at ? $activeSubscription->trial_ends_at->format('Y-m-d H:i:s') : 'N/A';
                            }),
                        Forms\Components\Placeholder::make('lemon_customer_id')
                            ->label('Lemon Squeezy Customer ID')
                            ->content(function ($record) {
                                if (!$record) return 'N/A'; // For create form
                                return $record->lemon_customer_id ?? 'N/A';
                            }),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->searchable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable(),
                Tables\Columns\IconColumn::make('subscription_status')
                    ->label('Subscribed')
                    ->boolean()
                    ->getStateUsing(function ($record) {
                        return $record->lemonSqueezySubscriptions()
                            ->where('status', 'active')
                            ->exists();
                    })
                    ->sortable(),
                Tables\Columns\TextColumn::make('current_plan')
                    ->label('Plan')
                    ->getStateUsing(function ($record) {
                        $subscription = $record->lemonSqueezySubscriptions()
                            ->where('status', 'active')
                            ->first();
                        
                        if ($subscription && $subscription->plan) {
                            return $subscription->plan->name;
                        }
                        
                        return null;
                    })
                    ->searchable(),
                Tables\Columns\TextColumn::make('email_verified_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit' => Pages\EditUser::route('/{record}/edit'),
        ];
    }
}
