<?php

namespace App\Filament\Resources;

use App\Filament\Resources\LemonSqueezySubscriptionResource\Pages;
use App\Filament\Resources\LemonSqueezySubscriptionResource\RelationManagers;
use App\Models\LemonSqueezySubscription;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class LemonSqueezySubscriptionResource extends Resource
{
    protected static ?string $model = LemonSqueezySubscription::class;

    protected static ?string $navigationIcon = 'heroicon-o-credit-card';
    
    protected static ?string $navigationGroup = 'Subscription Management';
    
    protected static ?string $navigationLabel = 'Subscriptions';
    
    protected static ?int $navigationSort = 2;
    
    public static function getNavigationBadge(): ?string
    {
        return static::getModel()::active()->count();
    }
    
    public static function getNavigationBadgeColor(): ?string
    {
        return 'success';
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Subscription Details')
                    ->description('Subscription information synchronized from Lemon Squeezy')
                    ->schema([
                        Forms\Components\TextInput::make('lemon_squeezy_id')
                            ->label('Subscription ID')
                            ->disabled(),
                        Forms\Components\TextInput::make('status')
                            ->disabled(),
                        Forms\Components\Select::make('billable_id')
                            ->relationship('billable', 'name')
                            ->label('User')
                            ->disabled(),
                        Forms\Components\TextInput::make('plan.name')
                            ->label('Plan')
                            ->disabled(),
                    ])->columns(2),
                
                Forms\Components\Section::make('Billing Information')
                    ->schema([
                        Forms\Components\TextInput::make('card_brand')
                            ->label('Card Type')
                            ->disabled(),
                        Forms\Components\TextInput::make('card_last_four')
                            ->label('Card Last Four')
                            ->disabled(),
                        Forms\Components\TextInput::make('product_id')
                            ->label('Product ID')
                            ->disabled(),
                        Forms\Components\TextInput::make('variant_id')
                            ->label('Price ID')
                            ->disabled(),
                    ])->columns(2),
                
                Forms\Components\Section::make('Subscription Dates')
                    ->schema([
                        Forms\Components\DateTimePicker::make('created_at')
                            ->label('Started At')
                            ->disabled(),
                        Forms\Components\DateTimePicker::make('renews_at')
                            ->label('Next Renewal')
                            ->disabled(),
                        Forms\Components\DateTimePicker::make('ends_at')
                            ->label('Expires At')
                            ->disabled(),
                        Forms\Components\DateTimePicker::make('trial_ends_at')
                            ->label('Trial Ends')
                            ->disabled(),
                    ])->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('billable.name')
                    ->label('User')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('plan.name')
                    ->label('Plan')
                    ->searchable()
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->badge()
                    ->color(fn (string $state): string => match ($state) {
                        'active' => 'success',
                        'past_due' => 'warning',
                        'unpaid' => 'danger',
                        'cancelled' => 'gray',
                        'expired' => 'gray',
                        default => 'gray',
                    })
                    ->searchable(),
                Tables\Columns\TextColumn::make('lemon_squeezy_id')
                    ->label('Subscription ID')
                    ->searchable()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('renews_at')
                    ->label('Next Renewal')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('ends_at')
                    ->label('Expires At')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Started At')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('trial_ends_at')
                    ->label('Trial Ends')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('card_brand')
                    ->label('Card')
                    ->formatStateUsing(fn ($state, $record) => $state ? $state . ' •••• ' . $record->card_last_four : null)
                    ->searchable()
                    ->toggleable(),
            ])
            ->defaultSort('ends_at', 'desc')
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        'active' => 'Active',
                        'past_due' => 'Past Due',
                        'unpaid' => 'Unpaid',
                        'cancelled' => 'Cancelled',
                        'expired' => 'Expired',
                    ])
                    ->default('active'),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
            ])
            ->bulkActions([]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListLemonSqueezySubscriptions::route('/'),
            'view' => Pages\ViewLemonSqueezySubscription::route('/{record}'),
        ];
    }
}
