<?php

use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Validation\Rule;
use Livewire\Attributes\Title;
use Livewire\Volt\Component;

new #[Title('Profile')] class extends Component {
    public string $name     = '';
    public string $email    = '';
    public string $company  = '';
    public string $phone_number  = '';
    public string $biography  = '';

    /**
     * Mount the component.
     */
    public function mount(): void
    {
        $this->name = Auth::user()->name;
        $this->email = Auth::user()->email;
        $this->company = Auth::user()->company;
        $this->phone_number = Auth::user()->phone_number;
        $this->biography = Auth::user()->biography;
    }

    /**
     * Update the profile information for the currently authenticated user.
     */
    public function updateProfileInformation(): void
    {
        $user = Auth::user();

        $validated = $this->validate([
            'name' => ['required', 'string', 'max:255'],

            'email' => [
                'required',
                'string',
                'lowercase',
                'email',
                'max:255',
                Rule::unique(User::class)->ignore($user->id)
            ],

            'company' => 'nullable|string|max:255',

            'phone_number' => 'nullable|phone:' . config('validation.phone_countries'),

            'biography' => 'nullable|string',
        ], [
            'phone_number.phone' => 'Please enter a valid phone number.',
        ]);

        $user->fill($validated);

        if ($user->isDirty('email')) {
            $user->email_verified_at = null;
        }

        $user->save();

        $this->dispatch('profile-updated', name: $user->name);
    }

    /**
     * Send an email verification notification to the current user.
     */
    public function resendVerificationNotification(): void
    {
        $user = Auth::user();

        if ($user->hasVerifiedEmail()) {
            $this->redirectIntended(default: route('dashboard', absolute: false));

            return;
        }

        $user->sendEmailVerificationNotification();

        Session::flash('status', 'verification-link-sent');
    }
}; ?>

<section class="w-full">
    @include('partials.settings-heading')

    <x-settings.layout :heading="__('Profile')" :subheading="__('Update your name and email address')">
        <form wire:submit="updateProfileInformation" class="my-6 w-full space-y-6">
            <flux:input wire:model="name" :label="__('First and Last Name')" type="text" required autofocus autocomplete="name" />

            <div>
                <flux:input wire:model="email" :label="__('Email')" type="email" required autocomplete="email" />

                @if (auth()->user() instanceof \Illuminate\Contracts\Auth\MustVerifyEmail &&! auth()->user()->hasVerifiedEmail())
                    <div>
                        <flux:text class="mt-4">
                            {{ __('Your email address is unverified.') }}

                            <flux:link class="text-sm cursor-pointer" wire:click.prevent="resendVerificationNotification">
                                {{ __('Click here to re-send the verification email.') }}
                            </flux:link>
                        </flux:text>

                        @if (session('status') === 'verification-link-sent')
                            <flux:text class="mt-2 font-medium !dark:text-green-400 !text-green-600">
                                {{ __('A new verification link has been sent to your email address.') }}
                            </flux:text>
                        @endif
                    </div>
                @endif
            </div>

            <flux:input wire:model="company" :label="__('Company')" type="text" autocomplete="company" />

            <flux:input wire:model="phone_number" :label="__('Phone Number')" type="text" autocomplete="phone_number" />

            @hasanyrole('superadmin|employee|admin')
            <flux:textarea wire:model="biography" :label="__('Biography')" type="text" autocomplete="biography" />
            @endhasanyrole

            <div class="flex items-center gap-4">
                <div class="flex items-center justify-end">
                    <flux:button  type="submit" class="!bg-indigo-700 hover:!bg-indigo-500 !text-white w-full">{{ __('Save') }}</flux:button>
                </div>

                <x-action-message class="me-3" on="profile-updated">
                    {{ __('Saved.') }}
                </x-action-message>
            </div>
        </form>

        @php
            $gdprEnabled = false;
            try {
                $extensionManager = app(\App\Extensions\ExtensionManager::class);
                $gdprEnabled = $extensionManager->isInstalled('gdpr-compliance')
                    && $extensionManager->isEnabled('gdpr-compliance');
            } catch (\Exception $e) {
                // Extension not available, use default behavior
            }
        @endphp

        @if($gdprEnabled)
            {{-- GDPR-compliant privacy and data management --}}
            <livewire:gdpr-compliance.profile-section />
        @else
            {{-- Standard account deletion --}}
            <livewire:settings.delete-user-form />
        @endif
    </x-settings.layout>
</section>
