<?php

use App\Models\Settings;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\{Auth, Session, Cache};
use Livewire\Attributes\Title;
use Livewire\Volt\Component;

new #[Title('License Management')] class extends Component {
    public string $license_key = '';
    public bool $saving = false;
    public $license_status = null;
    public $license = null;

    /**
     * Mount the component.
     */
    public function mount(): void
    {
        // Get license key from database or .env
        $rawKey = Settings::get('license_key') ?? config('ticaga.key') ?? '';

        // Check if the key is JSON encoded (from API response)
        if (!empty($rawKey) && str_starts_with(trim($rawKey), '{')) {
            try {
                $decoded = json_decode($rawKey, true);
                if (isset($decoded['key'])) {
                    $this->license_key = $decoded['key'];
                } else {
                    $this->license_key = $rawKey;
                }
            } catch (\Exception $e) {
                $this->license_key = $rawKey;
            }
        } else {
            $this->license_key = $rawKey;
        }

        $this->loadLicenseData();
    }

    /**
     * Load license data
     */
    public function loadLicenseData(): void
    {
        try {
            $this->license = Controller::resolve(true);
            $this->license_status = Controller::checkStatus();
        } catch (\Exception $e) {
            $this->license_status = [
                'status' => 'error',
                'msg' => 'Unable to validate license: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Update the license key.
     */
    public function updateLicenseKey(): void
    {
        $this->saving = true;

        $validated = $this->validate([
            'license_key' => ['required', 'string', 'min:10'],
        ]);

        try {
            // Save to database
            Settings::set('license_key', $this->license_key, 'string', 'Ticaga License Key');

            // Clear cache
            Controller::clearLicenseCache();

            // Reload license data
            $this->loadLicenseData();

            Session::flash('status', 'license-updated');
            $this->dispatch('license-updated');

        } catch (\Exception $e) {
            Session::flash('error', 'Failed to update license key: ' . $e->getMessage());
        } finally {
            $this->saving = false;
        }
    }

    /**
     * Refresh license status
     */
    public function refreshLicense(): void
    {
        Controller::clearLicenseCache();
        $this->loadLicenseData();
        Session::flash('status', 'license-refreshed');
    }

    /**
     * Get status badge color
     */
    public function getStatusColor(): string
    {
        return match ($this->license_status['status'] ?? 'unknown') {
            'active' => 'green',
            'Invalid License Key', 'No License Key', 'Blacklisted', 'Cancelled', 'Suspended', 'Invalid Location' => 'red',
            'Expired' => 'orange',
            default => 'gray',
        };
    }

    /**
     * Check if license is active
     */
    public function isActive(): bool
    {
        return isset($this->license_status['status']) && $this->license_status['status'] === 'active';
    }
}; ?>

<section class="w-full">
    @include('partials.settings-heading')

    <x-settings.layout
        :heading="__('License Management')"
        :subheading="__('Manage your Ticaga license key and view license information')"
        :fullWidth="true">

        <!-- License Information Card -->
        <div class="mb-8 bg-white border border-neutral-200 rounded-lg shadow-sm p-6 dark:bg-gray-800 dark:border-neutral-700">
            <div class="flex items-center justify-between mb-6">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white">
                    {{ __('License Information') }}
                </h3>

                <!-- Status Badge -->
                @if($this->license_status)
                    <flux:badge :color="$this->getStatusColor()" size="lg">
                        {{ ucfirst($this->license_status['status'] ?? 'Unknown') }}
                    </flux:badge>
                @endif
            </div>

            <!-- Error/Warning Messages -->
            @if($this->license_status && $this->license_status['status'] !== 'active')
                <div class="mb-6 p-4 text-sm text-red-800 bg-red-50 border border-red-300 rounded-lg dark:bg-gray-900 dark:text-red-400 dark:border-red-800" role="alert">
                    <div class="flex items-start">
                        <svg class="w-5 h-5 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clip-rule="evenodd"/>
                        </svg>
                        <div>
                            <span class="font-medium">{{ $this->license_status['msg'] ?? 'Unknown error occurred' }}</span>
                        </div>
                    </div>
                </div>
            @endif

            <!-- Update Available Notice -->
            @if($this->isActive() && isset($this->license['version']) && isset($this->license['latest_version']['stable']) && $this->license['version'] < $this->license['latest_version']['stable'])
                @if (isset($this->license['updates']) && $this->license['updates'] !== false && \Carbon\Carbon::parse($this->license['updates'])->isPast())
                    <div class="mb-6 p-4 text-sm text-red-800 bg-red-50 border border-red-300 rounded-lg dark:bg-gray-900 dark:text-red-400 dark:border-red-800" role="alert">
                        <div class="flex items-start">
                            <svg class="w-5 h-5 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                            </svg>
                            <div>
                                <span class="font-medium">{{ __('License Update Required') }}</span>
                                <p class="mt-1">
                                    {{ __('Your license is out of date. Version :latest is available. Please renew your license to continue receiving updates.', ['latest' => $this->license['latest_version']['stable']]) }}
                                </p>
                                <a href="https://billing.ticaga.com/client/login" target="_blank" class="mt-2 inline-block font-medium underline hover:no-underline">
                                    {{ __('Renew License') }}
                                </a>
                            </div>
                        </div>
                    </div>
                @else
                    <div class="mb-6 p-4 text-sm text-blue-800 bg-blue-50 border border-blue-300 rounded-lg dark:bg-gray-900 dark:text-blue-400 dark:border-blue-800" role="alert">
                        <div class="flex items-start">
                            <svg class="w-5 h-5 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clip-rule="evenodd"/>
                            </svg>
                            <div>
                                <span class="font-medium">{{ __('Update Available') }}</span>
                                <p class="mt-1">
                                    {{ __('Version :latest is available. You are using version :current.', ['latest' => $this->license['latest_version']['stable'], 'current' => $this->license['version']]) }}
                                </p>
                                <a href="https://ticaga.com/latest.zip" target="_blank" class="mt-2 inline-block font-medium underline hover:no-underline">
                                    {{ __('Download Update') }}
                                </a>
                            </div>
                        </div>
                    </div>
                @endif
            @endif

            <!-- License Details Grid -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                <div class="space-y-4">
                    <div>
                        <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('Status') }}</dt>
                        <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                            {{ ucfirst($this->license_status['status'] ?? 'Unknown') }}
                        </dd>
                    </div>

                    @if($this->isActive())
                        <div>
                            <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('License Type') }}</dt>
                            <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                                {{ $this->license['license_name'] ?? 'N/A' }}
                            </dd>
                        </div>
                    @endif
                </div>

                <div class="space-y-4">
                    <div>
                        <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('License Key') }}</dt>
                        <dd class="mt-1 text-sm text-gray-900 dark:text-white font-mono break-all">
                            {{ $this->license['label'] ?? substr($this->license_key, 0, 20) . '...' }}
                        </dd>
                    </div>

                    @if($this->isActive())
                        <div>
                            <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('Current Version') }}</dt>
                            <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                                {{ $this->license['version'] ?? 'N/A' }}
                            </dd>
                        </div>
                    @endif
                </div>

                <div class="space-y-4">
                    @if($this->isActive())
                        @if(isset($this->license['updates']) && $this->license['updates'] === false)
                            @if(isset($this->license['license_name']) && $this->license['license_name'] === 'Ticaga Lifetime')
                                <div>
                                    <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('Support Expires') }}</dt>
                                    <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                                        {{ __('Not Applicable') }}
                                    </dd>
                                </div>
                            @else
                                <div>
                                    <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('Renewal Date') }}</dt>
                                    <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                                        {{ isset($this->license['custom']['renew_date']) ? \Carbon\Carbon::parse($this->license['custom']['renew_date'])->format('jS F Y') : 'N/A' }}
                                    </dd>
                                </div>
                            @endif
                        @else
                            <div>
                                <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('Support Expires') }}</dt>
                                <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                                    {{ isset($this->license['updates']) ? \Carbon\Carbon::parse($this->license['updates'])->format('jS F Y') : 'N/A' }}
                                </dd>
                            </div>
                        @endif

                        @if(isset($this->license['latest_version']['stable']))
                            <div>
                                <dt class="text-sm font-medium text-gray-500 dark:text-gray-400">{{ __('Latest Version') }}</dt>
                                <dd class="mt-1 text-sm text-gray-900 dark:text-white">
                                    {{ $this->license['latest_version']['stable'] }}
                                </dd>
                            </div>
                        @endif
                    @endif
                </div>
            </div>

            <!-- Action Links -->
            @if($this->isActive())
                <div class="mt-6 pt-6 border-t border-gray-200 dark:border-gray-700">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-gray-700 dark:text-gray-300">{{ __('Manage Your License') }}</span>
                        <div class="flex gap-4">
                            @if(isset($this->license['custom']['license_type']) && $this->license['custom']['license_type'] === 'trial')
                                <a href="https://billing.ticaga.com/order" target="_blank" class="text-sm font-medium text-indigo-600 hover:text-indigo-500 dark:text-indigo-400 dark:hover:text-indigo-300">
                                    {{ __('Purchase License') }}
                                </a>
                            @elseif(isset($this->license['custom']['service_id']))
                                <a href="https://billing.ticaga.com/client/services/manage/{{ $this->license['custom']['service_id'] }}" target="_blank" class="text-sm font-medium text-indigo-600 hover:text-indigo-500 dark:text-indigo-400 dark:hover:text-indigo-300">
                                    {{ __('Manage Service') }}
                                </a>
                            @endif
                            <a href="https://billing.ticaga.com/client/login" target="_blank" class="text-sm font-medium text-indigo-600 hover:text-indigo-500 dark:text-indigo-400 dark:hover:text-indigo-300">
                                {{ __('Client Portal') }}
                            </a>
                        </div>
                    </div>
                </div>
            @endif
        </div>

        <!-- License Key Form -->
        <form wire:submit="updateLicenseKey" class="mb-8 bg-white border border-neutral-200 rounded-lg shadow-sm p-6 dark:bg-gray-800 dark:border-neutral-700">
            <h3 class="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                {{ __('Update License Key') }}
            </h3>

            <div class="space-y-4">
                <div>
                    <flux:input
                        wire:model="license_key"
                        :label="__('License Key')"
                        type="text"
                        required
                        autocomplete="off"
                        placeholder="Enter your Ticaga license key"
                    />
                    <flux:text class="mt-2 text-sm text-gray-600 dark:text-gray-400">
                        {{ __('Enter your license key from your Ticaga account. Changes are saved to the database.') }}
                    </flux:text>
                </div>

                <div class="flex items-center gap-4">
                    <flux:button type="submit" class="!bg-indigo-700 hover:!bg-indigo-500 !text-white" :disabled="$saving">
                        {{ $saving ? __('Saving...') : __('Save License Key') }}
                    </flux:button>

                    <flux:button type="button" wire:click="refreshLicense" variant="ghost">
                        {{ __('Refresh Status') }}
                    </flux:button>

                    @if (session('status') === 'license-updated')
                        <flux:text class="text-green-600 dark:text-green-400 font-medium">
                            {{ __('License key saved successfully!') }}
                        </flux:text>
                    @endif

                    @if (session('status') === 'license-refreshed')
                        <flux:text class="text-green-600 dark:text-green-400 font-medium">
                            {{ __('License status refreshed!') }}
                        </flux:text>
                    @endif

                    @if (session('error'))
                        <flux:text class="text-red-600 dark:text-red-400 font-medium">
                            {{ session('error') }}
                        </flux:text>
                    @endif
                </div>
            </div>
        </form>

        <!-- Help Text -->
        <div class="p-4 bg-blue-50 border border-blue-200 rounded-lg dark:bg-gray-900 dark:border-blue-900">
            <h4 class="text-sm font-medium text-blue-900 dark:text-blue-300 mb-2">
                {{ __('Need Help?') }}
            </h4>
            <p class="text-sm text-blue-800 dark:text-blue-400">
                {{ __('If you\'re having trouble with your license, please email our Accounts team at') }}
                <a href="mailto:accounts@ticaga.com" class="font-medium underline hover:no-underline">accounts@ticaga.com</a>
                {{ __('or visit our') }}
                <a href="https://docs.ticaga.com/" target="_blank" class="font-medium underline hover:no-underline">{{ __('documentation') }}</a>.
            </p>
        </div>
    </x-settings.layout>
</section>