<?php

use App\Extensions\Installed\Importer\Models\ImportJob;
use function Livewire\Volt\{state, mount, computed};

state(['activeJobs', 'completedJobs']);

mount(function () {
    $this->loadJobs();
});

$loadJobs = function () {
    $this->activeJobs = ImportJob::active()->latest()->get();
    $this->completedJobs = ImportJob::completed()->latest()->take(10)->get();
};

$deleteJob = function ($jobId) {
    $job = ImportJob::find($jobId);
    if ($job && in_array($job->status, ['completed', 'failed'])) {
        $job->delete();
        $this->loadJobs();
        session()->flash('message', 'Import job deleted successfully.');
    }
};

?>

<div>
    <div class="space-y-6 pb-16">
        <!-- Breadcrumb -->
        <nav class="flex text-sm text-gray-500 dark:text-gray-400" aria-label="Breadcrumb">
            <ol class="inline-flex items-center space-x-1 md:space-x-3">
                <li class="inline-flex items-center">
                    <a href="{{ route('extensions.importer.index') }}"
                       class="inline-flex items-center gap-1 text-gray-600 hover:text-indigo-500 dark:text-gray-300 dark:hover:text-indigo-400 transition">
                        <x-heroicon-o-arrow-uturn-left class="w-4 h-4" />
                        Importer
                    </a>
                </li>
                <li>
                    <div class="flex items-center">
                        <x-heroicon-s-chevron-right class="w-4 h-4 text-gray-400" />
                        <span class="ml-1 text-gray-500 dark:text-gray-400">Import Jobs</span>
                    </div>
                </li>
            </ol>
        </nav>

        <!-- Header Card -->
        <div class="bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-xl shadow-sm p-6">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                <div>
                    <h1 class="text-2xl font-semibold text-gray-900 dark:text-white">Import Jobs History</h1>
                    <p class="mt-2 text-sm text-gray-600 dark:text-gray-400 max-w-2xl">
                        Review active and completed imports. Jobs refresh automatically every few seconds.
                    </p>
                </div>
                <a href="{{ route('extensions.importer.index') }}"
                   class="inline-flex items-center px-4 py-2 rounded-xl bg-indigo-600 hover:bg-indigo-700 text-white font-semibold shadow-sm transition">
                    <x-heroicon-s-plus class="w-5 h-5 mr-2" />
                    New Import
                </a>
            </div>
        </div>

    @volt('extensions.importer.jobs.index')
    <div class="space-y-6" wire:poll.5s="loadJobs">

        @if (session('message'))
            <div class="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 text-green-800 dark:text-green-200 px-4 py-3 rounded">
                {{ session('message') }}
            </div>
        @endif

        <!-- Active Jobs -->
        @if($activeJobs->count() > 0)
        <div class="rounded-xl border border-gray-200 bg-white shadow-sm dark:border-gray-700 dark:bg-gray-900">
            <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-800/60">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Active Imports</h3>
            </div>
            <div class="divide-y divide-gray-200 dark:divide-gray-800/60">
                @foreach($activeJobs as $job)
                <div class="px-6 py-4 bg-white dark:bg-gray-900/50">
                    <div class="flex items-center justify-between mb-2">
                        <div class="flex-1">
                            <h4 class="text-base font-medium text-gray-900 dark:text-white">
                                {{ $job->name }}
                            </h4>
                            @php
                                $systemId = $job->options['database_system'] ?? ($job->source === 'supportpal' ? 'supportpal' : null);
                                if ($systemId) {
                                    try {
                                        $systemLabel = \App\Extensions\Installed\Importer\Services\DatabaseAdapters\DatabaseAdapterRegistry::resolve($systemId)::label();
                                    } catch (\Throwable $e) {
                                        $systemLabel = \Illuminate\Support\Str::headline($systemId);
                                    }
                                } else {
                                    $systemLabel = ucfirst($job->source);
                                }
                            @endphp
                            <p class="text-sm text-gray-500 dark:text-gray-400">
                                {{ $systemLabel }} • {{ ucfirst($job->type) }} • Started {{ $job->started_at?->diffForHumans() ?? 'Not started' }}
                            </p>
                        </div>
                        <div class="flex items-center space-x-4">
                            <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium
                                {{ $job->status === 'processing' ? 'bg-blue-100 text-blue-800 dark:bg-blue-900/20 dark:text-blue-400' : 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/20 dark:text-yellow-400' }}">
                                {{ ucfirst($job->status) }}
                            </span>
                            @php
                                $jobRoute = in_array($job->source, ['database', 'supportpal'], true)
                                    ? route('extensions.importer.database.process', $job)
                                    : route('extensions.importer.csv.process', ['jobId' => $job->id]);
                            @endphp
                            <a href="{{ $jobRoute }}"
                               class="text-indigo-600 hover:text-indigo-700 dark:text-indigo-300 font-medium text-sm">
                                View Details
                            </a>
                        </div>
                    </div>
                    <div class="mt-2">
                        <div class="flex items-center justify-between text-sm text-gray-600 dark:text-gray-400 mb-1">
                            <span>{{ number_format($job->processed_rows) }} / {{ number_format($job->total_rows) }} rows</span>
                            <div class="flex items-center gap-4">
                                @if($job->etaForHumans())
                                    <span class="text-xs md:text-sm text-gray-500 dark:text-gray-400">ETA: {{ $job->etaForHumans() }}</span>
                                @endif
                                <span>{{ number_format($job->getProgressPercentage(), 1) }}%</span>
                            </div>
                        </div>
                        <div class="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2">
                            <div class="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                 style="width: {{ $job->getProgressPercentage() }}%"></div>
                        </div>
                    </div>
                </div>
                @endforeach
            </div>
        </div>
        @else
        <div class="rounded-xl border border-gray-200 bg-white p-6 text-center shadow-sm dark:border-gray-700 dark:bg-gray-900">
            <x-heroicon-o-clock class="mx-auto h-12 w-12 text-gray-400 mb-4" />
            <h3 class="text-lg font-medium text-gray-900 dark:text-white mb-2">No Active Imports</h3>
            <p class="text-gray-600 dark:text-gray-400">Start a new import to see it here.</p>
        </div>
        @endif

        <!-- Completed Jobs -->
        @if($completedJobs->count() > 0)
        <div class="rounded-xl border border-gray-200 bg-white shadow-sm dark:border-gray-700 dark:bg-gray-900">
            <div class="px-6 py-4 border-b border-gray-200 dark:border-gray-800/60">
                <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Recent Completed Imports</h3>
            </div>
            <div class="divide-y divide-gray-200 dark:divide-gray-800/60">
                @foreach($completedJobs as $job)
                <div class="px-6 py-4 bg-white dark:bg-gray-900/50">
                    <div class="flex items-center justify-between">
                        <div class="flex-1">
                            <h4 class="text-base font-medium text-gray-900 dark:text-white">
                                {{ $job->name }}
                            </h4>
                            <p class="text-sm text-gray-500 dark:text-gray-400">
                                {{ ucfirst($job->source) }} • {{ ucfirst($job->type) }} • Completed {{ $job->completed_at?->diffForHumans() }}
                            </p>
                            <p class="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                <span class="text-green-600 dark:text-green-400">{{ number_format($job->processed_rows) }} imported</span>
                                @if($job->failed_rows > 0)
                                    • <span class="text-red-600 dark:text-red-400">{{ number_format($job->failed_rows) }} failed</span>
                                @endif
                            </p>
                        </div>
                        <div class="flex items-center space-x-4">
                            <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-emerald-100 text-emerald-700 dark:bg-emerald-900/20 dark:text-emerald-300">
                                Completed
                            </span>
                            <button wire:click="deleteJob({{ $job->id }})"
                                    class="text-red-600 hover:text-red-700 dark:text-red-400 text-sm"
                                    onclick="return confirm('Are you sure you want to delete this import job?')">
                                Delete
                            </button>
                        </div>
                    </div>
                </div>
                @endforeach
            </div>
        </div>
        @endif
    </div>
    @endvolt
    </div>
</div>
