<?php

namespace App\Services\Installation;

use Illuminate\Database\Connection;
use Illuminate\Support\Facades\DB;
use PDO;
use RuntimeException;
use Symfony\Component\Process\Process;

class InstallationService
{
    private const TEST_CONNECTION = '__installer__';

    private const SESSION_CONNECTION = '__installer_preview__';

    protected bool $connectionChecked = false;

    protected bool $connectionAvailable = false;

    protected ?string $connectionError = null;

    /**
     * Determine if the application server meets the baseline requirements.
     */
    public function systemRequirements(): array
    {
        $requirements = [];

        $requirements[] = $this->formatRequirement(
            'PHP 8.2 or higher',
            version_compare(PHP_VERSION, '8.2.0', '>='),
            'Minimum: 8.2.0 — Current: ' . PHP_VERSION,
        );

        $requirements[] = $this->formatRequirement(
            'ionCube Loader installed',
            $this->ioncubeLoaderSatisfied(),
            $this->ioncubeLoaderDetails(),
        );

        $requirements[] = $this->formatRequirement(
            'Composer available',
            $this->composerAvailable(),
            $this->composerAvailable()
                ? 'Composer binary detected'
                : 'Install Composer globally or set COMPOSER_BINARY',
        );

        $requirements[] = $this->formatRequirement(
            'Laravel 12.x',
            $this->laravelVersionSatisfied(),
            'Current version: ' . app()->version(),
            blocking: false,
        );

        $requirements[] = $this->formatRequirement(
            'Let\'s Encrypt (certbot) available',
            $this->certbotAvailable(),
            $this->certbotAvailable()
                ? 'certbot binary detected'
                : 'Install certbot to automate SSL certificates',
            blocking: false,
        );

        foreach ($this->requiredPhpExtensions() as $label => $extension) {
            $loaded = extension_loaded($extension);

            $requirements[] = $this->formatRequirement(
                sprintf('%s PHP extension', $label),
                $loaded,
                $loaded
                    ? 'Extension detected'
                    : 'Enable the extension in php.ini',
            );
        }

        $requirements[] = $this->formatRequirement(
            'Writable storage directory',
            is_writable(storage_path()),
            storage_path(),
        );

        $requirements[] = $this->formatRequirement(
            'Writable bootstrap/cache directory',
            is_writable(base_path('bootstrap/cache')),
            base_path('bootstrap/cache'),
        );

        $databaseStatus = $this->databaseRequirementStatus();

        $requirements[] = $this->formatRequirement(
            'MySQL or MariaDB available',
            $databaseStatus['passed'],
            $databaseStatus['details'],
            blocking: false,
            pending: $databaseStatus['status'] === 'pending',
            status: $databaseStatus['status'],
        );

        return $requirements;
    }

    public function requirementsSatisfied(?array $requirements = null): bool
    {
        $requirements ??= $this->systemRequirements();

        foreach ($requirements as $requirement) {
            if (($requirement['blocking'] ?? true) && ! ($requirement['passed'] ?? false)) {
                return false;
            }
        }

        return true;
    }

    public function dependencyStatus(bool $performCheck = true): array
    {
        $dependencies = [];

        foreach ($this->dependencyDefinitions() as $definition) {
            if ($performCheck) {
                $installed = ($definition['checker'])();

                $dependencies[] = [
                    'label' => $definition['label'],
                    'passed' => $installed,
                    'details' => $installed
                        ? $definition['installed_details']
                        : $definition['missing_details'],
                    'status' => $installed ? 'passed' : 'failed',
                ];

                continue;
            }

            $dependencies[] = [
                'label' => $definition['label'],
                'passed' => false,
                'details' => 'Check pending until you review this step.',
                'status' => 'pending',
            ];
        }

        return $dependencies;
    }

    public function dependenciesSatisfied(?array $dependencies = null): bool
    {
        $dependencies ??= $this->dependencyStatus();

        foreach ($dependencies as $dependency) {
            if (! ($dependency['passed'] ?? false)) {
                return false;
            }
        }

        return true;
    }

    public function pendingDependencyStatus(): array
    {
        return $this->dependencyStatus(performCheck: false);
    }

    protected function dependencyDefinitions(): array
    {
        return [
            [
                'label' => 'Composer dependencies',
                'checker' => fn () => $this->composerDependenciesInstalled(),
                'installed_details' => 'vendor/autoload.php detected',
                'missing_details' => 'Composer install required',
            ],
            [
                'label' => 'Node dependencies',
                'checker' => fn () => $this->nodeDependenciesInstalled(),
                'installed_details' => 'node_modules directory detected',
                'missing_details' => 'npm install required for asset builds',
            ],
        ];
    }

    public function summarizeRequirements(array $requirements): array
    {
        $collection = collect($requirements);

        $hasBlockingFailures = $collection->contains(
            fn ($requirement) => ($requirement['blocking'] ?? true) && ! ($requirement['passed'] ?? false)
        );

        $hasWarnings = $collection->contains(function ($requirement) {
            if ($requirement['passed'] ?? false) {
                return false;
            }

            if (($requirement['blocking'] ?? true) === false) {
                return true;
            }

            return ($requirement['status'] ?? null) === 'warning';
        });

        $hasPending = $collection->contains(fn ($requirement) => ($requirement['status'] ?? null) === 'pending');

        $issues = $collection
            ->filter(fn ($requirement) => ! ($requirement['passed'] ?? false) || in_array($requirement['status'] ?? null, ['warning', 'pending'], true))
            ->count();

        $level = 'success';

        if ($hasBlockingFailures) {
            $level = 'error';
        } elseif ($hasWarnings) {
            $level = 'warning';
        } elseif ($hasPending) {
            $level = 'pending';
        }

        return [
            'level' => $level,
            'open' => $hasBlockingFailures || $hasWarnings || $hasPending,
            'issues' => $issues,
        ];
    }

    public function summarizeDependencies(array $dependencies): array
    {
        $missingDependencies = collect($dependencies)
            ->reject(fn ($dependency) => $dependency['passed'] ?? false)
            ->count();

        return [
            'level' => $missingDependencies > 0 ? 'warning' : 'success',
            'open' => $missingDependencies > 0,
            'issues' => $missingDependencies,
        ];
    }

    public function summarizeDatabaseReadiness(bool $connection, bool $hasMigrations, bool $hasUsersTable, bool $hasUsers): array
    {
        $issues = 0;
        $level = 'success';

        if (! $connection) {
            $issues++;
            $level = 'error';
        }

        $schemaConcerns = collect([
            $hasMigrations,
            $hasUsersTable,
            $hasUsers,
        ])->filter(fn ($value) => $value === false)->count();

        $issues += $schemaConcerns;

        if ($level !== 'error' && $schemaConcerns > 0) {
            $level = 'warning';
        }

        return [
            'level' => $level,
            'open' => $level !== 'success',
            'issues' => $issues,
        ];
    }

    /**
     * Determine if the application requires installation.
     */
    public function needsInstallation(): bool
    {
        // Check for installation lock file first - this prevents re-running the installer
        if ($this->isInstalled()) {
            return false;
        }

        if (! $this->databaseIsReachable()) {
            return true;
        }

        if (! $this->hasMigrationsTable()) {
            return true;
        }

        if (! $this->hasUsersTable()) {
            return true;
        }

        if (! $this->hasUsers()) {
            return true;
        }

        return false;
    }

    /**
     * Check if the installation has been completed (lock file exists).
     */
    public function isInstalled(): bool
    {
        return file_exists($this->getInstallLockPath());
    }

    /**
     * Get the path to the installation lock file.
     */
    protected function getInstallLockPath(): string
    {
        return storage_path('installed.lock');
    }

    /**
     * Check whether the database connection can be established.
     */
    public function databaseIsReachable(): bool
    {
        if (! $this->connectionChecked) {
            try {
                $this->withDatabaseConnection(function (Connection $connection): void {
                    $connection->getPdo();
                });

                $this->connectionAvailable = true;
            } catch (\Throwable $exception) {
                $this->connectionError = $exception->getMessage();
                $this->connectionAvailable = false;
            }

            $this->connectionChecked = true;
        }

        return $this->connectionAvailable;
    }

    public function resetConnectionCheck(): void
    {
        $this->connectionChecked = false;
        $this->connectionAvailable = false;
        $this->connectionError = null;
    }

    /**
     * Determine if the migrations table exists.
     */
    public function hasMigrationsTable(): bool
    {
        if (! $this->databaseIsReachable()) {
            return false;
        }

        try {
            return (bool) $this->withDatabaseConnection(function (Connection $connection) {
                return $connection->getSchemaBuilder()->hasTable('migrations');
            });
        } catch (\Throwable $exception) {
            $this->connectionError = $exception->getMessage();

            return false;
        }
    }

    /**
     * Determine if the users table exists.
     */
    public function hasUsersTable(): bool
    {
        if (! $this->databaseIsReachable()) {
            return false;
        }

        try {
            return (bool) $this->withDatabaseConnection(function (Connection $connection) {
                return $connection->getSchemaBuilder()->hasTable('users');
            });
        } catch (\Throwable $exception) {
            $this->connectionError = $exception->getMessage();

            return false;
        }
    }

    /**
     * Determine whether any user accounts have been created.
     */
    public function hasUsers(): bool
    {
        if (! $this->hasUsersTable()) {
            return false;
        }

        try {
            return (bool) $this->withDatabaseConnection(function (Connection $connection) {
                return $connection->table('users')->exists();
            });
        } catch (\Throwable $exception) {
            $this->connectionError = $exception->getMessage();

            return false;
        }
    }

    /**
     * Retrieve the most recent connection error, if any.
     */
    public function getConnectionError(): ?string
    {
        return $this->connectionError;
    }

    public function progressSteps(): array
    {
        $steps = [
            [
                'key' => 'requirements',
                'label' => 'System requirements',
                'completed' => $this->requirementsSatisfied(),
                'current' => false,
            ],
            [
                'key' => 'configuration',
                'label' => 'Database configuration',
                'completed' => $this->databaseIsReachable(),
                'current' => false,
            ],
            [
                'key' => 'schema',
                'label' => 'Database setup',
                'completed' => $this->hasMigrationsTable() && $this->hasUsersTable(),
                'current' => false,
            ],
            [
                'key' => 'administrator',
                'label' => 'Administrator account',
                'completed' => $this->hasUsers(),
                'current' => false,
            ],
        ];

        $currentIndex = 0;

        foreach ($steps as $index => $step) {
            if (! $step['completed']) {
                $currentIndex = $index;
                break;
            }

            $currentIndex = $index;
        }

        foreach ($steps as $index => &$step) {
            $step['current'] = $index === $currentIndex && ! $step['completed'];
        }

        if ($this->requirementsSatisfied() && ! $this->databaseIsReachable()) {
            $steps[1]['current'] = true;
        }

        return $steps;
    }

    public function progressPercentage(): int
    {
        $steps = $this->progressSteps();
        $completed = 0;

        foreach ($steps as $step) {
            if ($step['completed']) {
                $completed++;
            }
        }

        return (int) round(($completed / max(count($steps), 1)) * 100);
    }

    public function testDatabaseConnection(array $configuration): void
    {
        $tempConnection = $this->buildTemporaryConnection($configuration);

        config(['database.connections.' . self::TEST_CONNECTION => $tempConnection]);

        try {
            DB::purge(self::TEST_CONNECTION);
            DB::connection(self::TEST_CONNECTION)->getPdo();
        } catch (\Throwable $exception) {
            throw new RuntimeException($exception->getMessage(), $exception->getCode(), $exception);
        } finally {
            DB::disconnect(self::TEST_CONNECTION);
            config()->offsetUnset('database.connections.' . self::TEST_CONNECTION);
        }
    }

    public function updateEnvironmentFile(array $values): void
    {
        $path = base_path('.env');

        if (! file_exists($path)) {
            throw new \RuntimeException('.env file not found. Please ensure the .env file exists before running the installer.');
        }

        // Create backup of .env file before modifying (in storage directory for proper permissions)
        $backupPath = storage_path('app/.env.backup.' . now()->format('Y-m-d_His'));
        if (! copy($path, $backupPath)) {
            throw new \RuntimeException('Failed to create backup of .env file at: ' . $backupPath);
        }

        $content = file_get_contents($path);

        // Ensure .env file is writable by the web server
        if (! is_writable($path)) {
            // Try to make it writable
            @chmod($path, 0664);

            if (! is_writable($path)) {
                throw new \RuntimeException(
                    'The .env file is not writable by the web server. ' .
                    'Please run: sudo chmod 664 ' . $path . ' && sudo chown apache:apache ' . $path
                );
            }
        }

        foreach ($values as $key => $value) {
            $formatted = $this->formatEnvValue($value);
            $pattern = '/^' . preg_quote($key, '/') . '=.*$/m';

            if (preg_match($pattern, $content)) {
                $content = preg_replace($pattern, $key . '=' . $formatted, $content);
            } else {
                $content .= PHP_EOL . $key . '=' . $formatted;
            }
        }

        // Validate write operation
        if (file_put_contents($path, $content) === false) {
            // Restore from backup if write failed
            copy($backupPath, $path);
            throw new \RuntimeException('Failed to update .env file. Backup has been restored.');
        }
    }

    public function markInstallationComplete(): void
    {
        // Create installation lock file to prevent re-running the installer
        $lockPath = $this->getInstallLockPath();
        $lockData = json_encode([
            'version' => config('ticaga.version', '1.0.0'),
            'installed_at' => now()->toIso8601String(),
            'last_upgraded_at' => now()->toIso8601String(),
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
        ], JSON_PRETTY_PRINT);

        file_put_contents($lockPath, $lockData);

        $this->resetConnectionCheck();
    }

    public function ensureDependenciesInstalled(): void
    {
        if (! $this->composerDependenciesInstalled()) {
            $this->runComposerInstall();
        }

        if (file_exists(base_path('package.json')) && ! $this->nodeDependenciesInstalled()) {
            $this->runNpmInstall();
        }
    }

    protected function formatEnvValue(mixed $value): string
    {
        if ($value === null) {
            return '';
        }

        if (is_bool($value)) {
            return $value ? 'true' : 'false';
        }

        $value = (string) $value;

        if (preg_match('/\s/', $value)) {
            return '"' . addcslashes($value, '"') . '"';
        }

        return $value;
    }

    protected function composerDependenciesInstalled(): bool
    {
        return file_exists(base_path('vendor/autoload.php'));
    }

    protected function nodeDependenciesInstalled(): bool
    {
        return is_dir(base_path('node_modules'));
    }

    protected function runComposerInstall(): void
    {
        $command = $this->findComposerBinary();

        if ($command === null) {
            throw new RuntimeException('Composer is not available to install PHP dependencies.');
        }

        $process = Process::fromShellCommandline($command . ' install --no-dev --optimize-autoloader --no-interaction', base_path());
        $process->setTimeout(300);
        $process->run();

        if (! $process->isSuccessful()) {
            throw new RuntimeException(trim($process->getErrorOutput() ?: $process->getOutput()));
        }
    }

    protected function runNpmInstall(): void
    {
        $command = $this->findNpmBinary();

        if ($command === null) {
            throw new RuntimeException('npm is not available to install frontend dependencies.');
        }

        $process = Process::fromShellCommandline($command . ' install --production', base_path());
        $process->setTimeout(300);
        $process->run();

        if (! $process->isSuccessful()) {
            throw new RuntimeException(trim($process->getErrorOutput() ?: $process->getOutput()));
        }
    }

    protected function findComposerBinary(): ?string
    {
        static $cached = false;
        static $binary = null;

        if ($cached) {
            return $binary;
        }

        $cached = true;

        $configured = env('COMPOSER_BINARY');

        if (! empty($configured)) {
            $binary = $configured;

            return $binary;
        }

        $candidates = [
            'composer',
            '/usr/local/bin/composer',
            '/usr/bin/composer',
        ];

        foreach ($candidates as $candidate) {
            $process = Process::fromShellCommandline($candidate . ' --version');
            $process->run();

            if ($process->isSuccessful()) {
                $binary = $candidate;

                return $binary;
            }
        }

        if (file_exists(base_path('composer.phar'))) {
            $binary = PHP_BINARY . ' ' . base_path('composer.phar');

            return $binary;
        }

        return null;
    }

    protected function findNpmBinary(): ?string
    {
        static $cached = false;
        static $binary = null;

        if ($cached) {
            return $binary;
        }

        $cached = true;

        $candidates = [
            env('NPM_BINARY'),
            'npm',
            '/usr/local/bin/npm',
            '/usr/bin/npm',
        ];

        foreach ($candidates as $candidate) {
            if (empty($candidate)) {
                continue;
            }

            $process = Process::fromShellCommandline($candidate . ' --version');
            $process->run();

            if ($process->isSuccessful()) {
                $binary = $candidate;

                return $binary;
            }
        }

        return null;
    }

    protected function formatRequirement(
        string $label,
        bool $passed,
        string $details,
        bool $blocking = true,
        bool $pending = false,
        ?string $status = null,
    ): array {
        $status ??= $passed
            ? 'passed'
            : ($pending ? 'pending' : ($blocking ? 'failed' : 'warning'));

        return [
            'label' => $label,
            'passed' => $passed,
            'details' => $details,
            'blocking' => $blocking,
            'pending' => $pending,
            'status' => $status,
        ];
    }

    protected function ioncubeLoaderSatisfied(): bool
    {
        return $this->ioncubeLoaderAvailable();
    }

    protected function ioncubeLoaderDetails(): string
    {
        if (! $this->ioncubeLoaderAvailable()) {
            return 'Install the ionCube Loader extension';
        }

        $version = $this->ioncubeLoaderVersion();

        if ($version === null) {
            return 'ionCube Loader detected';
        }

        return 'ionCube Loader detected — version ' . $version;
    }

    protected function ioncubeLoaderAvailable(): bool
    {
        return extension_loaded('ionCube Loader') || extension_loaded('ioncube_loader');
    }

    protected function ioncubeLoaderVersion(): ?string
    {
        if (function_exists('ioncube_loader_version')) {
            $version = ioncube_loader_version();

            if (is_int($version)) {
                $version = (string) $version;
            }

            if (is_string($version)) {
                $version = trim($version);

                if (preg_match('/^(\d+(?:\.\d+)+)/', $version, $matches)) {
                    return $matches[1];
                }
            }
        }

        if (function_exists('ioncube_loader_version_array')) {
            $version = ioncube_loader_version_array();

            if (is_array($version) && isset($version['version'])) {
                return $version['version'];
            }
        }

        return null;
    }

    protected function composerAvailable(): bool
    {
        return $this->findComposerBinary() !== null;
    }

    protected function laravelVersionSatisfied(): bool
    {
        return str_starts_with(app()->version(), '12.');
    }

    protected function certbotAvailable(): bool
    {
        $commands = [
            'command -v certbot',
            'which certbot',
        ];

        foreach ($commands as $command) {
            $process = Process::fromShellCommandline($command);
            $process->run();

            if ($process->isSuccessful()) {
                return true;
            }
        }

        return false;
    }

    protected function requiredPhpExtensions(): array
    {
        return [
            'BCMath' => 'bcmath',
            'Ctype' => 'ctype',
            'cURL' => 'curl',
            'DOM' => 'dom',
            'Fileinfo' => 'fileinfo',
            'JSON' => 'json',
            'Mbstring' => 'mbstring',
            'OpenSSL' => 'openssl',
            'PDO' => 'pdo',
            'Tokenizer' => 'tokenizer',
            'XML' => 'xml',
        ];
    }

    protected function databaseRequirementStatus(): array
    {
        if (! $this->databaseIsReachable()) {
            return [
                'passed' => false,
                'details' => 'Version check pending — configure the database connection below.',
                'status' => 'pending',
            ];
        }

        $versionString = $this->detectDatabaseVersion();

        if ($versionString === null) {
            return [
                'passed' => false,
                'details' => 'Connected, but unable to determine database engine automatically.',
                'status' => 'warning',
            ];
        }

        $lowerVersion = strtolower($versionString);
        $isMariaDb = str_contains($lowerVersion, 'mariadb');
        $isMySql = str_contains($lowerVersion, 'mysql');

        if (! $isMariaDb && ! $isMySql) {
            return [
                'passed' => false,
                'details' => 'Detected database engine: ' . $versionString,
                'status' => 'warning',
            ];
        }

        $numericVersion = $this->extractVersionNumber($versionString);

        $details = $numericVersion === null
            ? sprintf('Detected %s', $isMariaDb ? 'MariaDB' : 'MySQL')
            : sprintf('Detected %s version %s', $isMariaDb ? 'MariaDB' : 'MySQL', $numericVersion);

        return [
            'passed' => true,
            'details' => $details,
            'status' => 'passed',
        ];
    }

    protected function detectDatabaseVersion(): ?string
    {
        try {
            $result = $this->withDatabaseConnection(function (Connection $connection) {
                return $connection->select('select version() as version');
            });
        } catch (\Throwable $exception) {
            $this->connectionError = $exception->getMessage();

            return null;
        }

        if (! isset($result[0])) {
            return null;
        }

        $row = (array) $result[0];

        return $row['version'] ?? null;
    }

    protected function extractVersionNumber(string $version): ?string
    {
        if (preg_match('/(\d+\.\d+(?:\.\d+)?)/', $version, $matches)) {
            return $matches[1];
        }

        return null;
    }

    protected function withDatabaseConnection(callable $callback)
    {
        $sessionConfiguration = $this->sessionDatabaseConfiguration();

        if ($sessionConfiguration !== null) {
            return $this->usingTemporaryConnection($sessionConfiguration, self::SESSION_CONNECTION, $callback);
        }

        return $callback(DB::connection());
    }

    protected function usingTemporaryConnection(array $configuration, string $name, callable $callback)
    {
        $temporary = $this->buildTemporaryConnection($configuration);

        config(['database.connections.' . $name => $temporary]);

        try {
            DB::purge($name);

            return $callback(DB::connection($name));
        } finally {
            DB::disconnect($name);
            config()->offsetUnset('database.connections.' . $name);
        }
    }

    protected function buildTemporaryConnection(array $configuration): array
    {
        $base = config('database.connections.mysql', []);

        $options = $base['options'] ?? [];
        $options[PDO::ATTR_TIMEOUT] = $options[PDO::ATTR_TIMEOUT] ?? 5;

        return array_merge($base, [
            'driver' => 'mysql',
            'host' => $configuration['host'] ?? '127.0.0.1',
            'port' => (string) ($configuration['port'] ?? '3306'),
            'database' => $configuration['database'] ?? '',
            'username' => $configuration['username'] ?? '',
            'password' => $this->normalizeDatabasePassword($configuration),
            'options' => $options,
        ]);
    }

    protected function normalizeDatabasePassword(array $configuration): string
    {
        if (! array_key_exists('password', $configuration)) {
            return '';
        }

        $password = $configuration['password'];

        if ($password === null) {
            return '';
        }

        return (string) $password;
    }

    protected function sessionDatabaseConfiguration(): ?array
    {
        if (! app()->bound('session')) {
            return null;
        }

        try {
            $configuration = session('install.database');
        } catch (\Throwable $exception) {
            return null;
        }

        return is_array($configuration) ? $configuration : null;
    }
}