<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class MailAccount extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'imap_host',
        'imap_port',
        'imap_encryption',
        'imap_validate_cert',
        'imap_username',
        'imap_password',
        'imap_protocol',
        'smtp_host',
        'smtp_port',
        'smtp_encryption',
        'smtp_username',
        'smtp_password',
        'is_active',
        'is_default',
        'from_name',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'imap_port' => 'integer',
        'smtp_port' => 'integer',
        'imap_validate_cert' => 'boolean',
        'is_active' => 'boolean',
        'is_default' => 'boolean',
        'imap_password' => 'encrypted',
        'smtp_password' => 'encrypted',
    ];

    /**
     * The attributes that should be hidden.
     *
     * @var list<string>
     */
    protected $hidden = [
        'imap_password',
        'smtp_password',
    ];

    /**
     * Get the departments using this mail account.
     */
    public function departments(): HasMany
    {
        return $this->hasMany(Departments::class);
    }

    /**
     * Scope a query to only include active mail accounts.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to get the default mail account.
     */
    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    /**
     * Get IMAP configuration as array for webklex/laravel-imap.
     */
    public function getImapConfigArray(): array
    {
        return [
            'host' => $this->imap_host,
            'port' => $this->imap_port,
            'protocol' => $this->imap_protocol ?? 'imap',
            'encryption' => $this->imap_encryption ?? 'ssl',
            'validate_cert' => $this->imap_validate_cert ?? true,
            'username' => $this->imap_username,
            'password' => $this->imap_password,
            'authentication' => null,
            'rfc' => 'RFC822',
            'timeout' => 30,
            'extensions' => [],
        ];
    }

    /**
     * Get SMTP configuration as array for Laravel mailer.
     */
    public function getSmtpConfigArray(): array
    {
        return [
            'transport' => 'smtp',
            'host' => $this->smtp_host,
            'port' => $this->smtp_port,
            'username' => $this->smtp_username,
            'password' => $this->smtp_password,
            'encryption' => $this->smtp_encryption,
            'from' => [
                'address' => $this->email,
                'name' => $this->from_name ?? $this->name,
            ],
        ];
    }

    /**
     * Set the account as default and unset all others.
     */
    public function setAsDefault(): void
    {
        static::where('id', '!=', $this->id)->update(['is_default' => false]);
        $this->update(['is_default' => true]);
    }
}
