<?php

namespace App\Livewire\Settings\Categories;

use App\Models\Categories;
use App\Enums\Type;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Layout;

#[Layout('components.layouts.app', ['title' => 'Categories'])]
class Index extends Component
{
    use WithPagination;

    public $search = '';
    public $typeFilter = ''; // Filter by category type (knowledgebase, announcements, etc.)
    public $showDeleteModal = false;
    public $categoryToDelete = null;

    protected $queryString = ['search', 'typeFilter'];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingTypeFilter()
    {
        $this->resetPage();
    }

    public function confirmDelete($categoryId)
    {
        $this->categoryToDelete = $categoryId;
        $this->showDeleteModal = true;
    }

    public function cancelDelete()
    {
        $this->showDeleteModal = false;
        $this->categoryToDelete = null;
    }

    public function deleteCategory()
    {
        $category = Categories::find($this->categoryToDelete);

        if ($category) {
            // Convert enum to string value for comparison
            $typeValue = $category->type instanceof \BackedEnum ? $category->type->value : $category->type;

            // Check if category has content (articles or announcements)
            $contentCount = 0;

            if ($typeValue === 'knowledgebase') {
                // Only check if Knowledge Base extension is installed
                $kbExtension = \App\Models\Extension::where('extension_id', 'knowledgebase')
                    ->where('enabled', true)
                    ->first();

                if ($kbExtension && class_exists('\App\Extensions\Installed\Knowledgebase\Models\Article')) {
                    $contentCount = $category->knowledgebase()->count();
                }
            } elseif ($typeValue === 'announcements') {
                $contentCount = $category->announcements()->count();
            }

            // Check if category has sub-categories
            $childCount = $category->children()->count();

            if ($contentCount > 0) {
                $contentType = $typeValue === 'knowledgebase' ? 'article' : 'announcement';
                session()->flash('error', "Cannot delete category with {$contentCount} {$contentType}" . ($contentCount !== 1 ? 's' : '') . ". Please reassign or delete the content first.");
            } elseif ($childCount > 0) {
                session()->flash('error', "Cannot delete category with {$childCount} sub-categor" . ($childCount === 1 ? 'y' : 'ies') . ". Please delete or reassign the sub-categories first.");
            } else {
                $category->delete();
                session()->flash('success', 'Category deleted successfully.');
            }
        }

        $this->showDeleteModal = false;
        $this->categoryToDelete = null;
    }

    public function render()
    {
        if ($this->search) {
            // When searching, show flat list with breadcrumb paths
            $query = Categories::query()
                ->where(function ($q) {
                    $q->where('category_name', 'like', '%' . $this->search . '%')
                      ->orWhere('uri', 'like', '%' . $this->search . '%');
                })
                ->with(['parent', 'children']);

            // Apply type filter
            if ($this->typeFilter) {
                $query->where('type', $this->typeFilter);
            }

            $categories = $query->orderBy('category_name')->paginate(15);
        } else {
            // When not searching, show hierarchical tree view (top-level only, with children loaded)
            $query = Categories::query()
                ->topLevel()
                ->with(['children.children'])
                ->ordered();

            // Apply type filter
            if ($this->typeFilter) {
                $query->where('type', $this->typeFilter);
            }

            $categories = $query->get();
        }

        // Get available types for filter dropdown
        $availableTypes = Categories::distinct()->pluck('type')->map(function ($type) {
            // Convert enum to string value if it's an enum
            $typeValue = $type instanceof \BackedEnum ? $type->value : $type;

            // Ensure we have a string for str_replace
            $typeString = (string) $typeValue;

            return [
                'value' => $typeValue,
                'label' => ucfirst(str_replace('_', ' ', $typeString))
            ];
        });

        return view('livewire.settings.categories.index', [
            'categories' => $categories,
            'availableTypes' => $availableTypes,
            'typeFilter' => $this->typeFilter,
            'search' => $this->search,
        ]);
    }
}
