<?php

namespace App\Livewire\Settings\Categories;

use App\Models\Categories;
use App\Enums\Type;
use Livewire\Component;
use Illuminate\Support\Str;
use Livewire\Attributes\Layout;

#[Layout("components.layouts.app", ["title" => "Edit Category"])]
class Edit extends Component
{
    public $category;
    public $category_name = "";
    public $uri = "";
    public $type = "";
    public $display = 1;
    public $auto_generate_uri = false;
    public $content_count = 0;
    public $parent_id = null;
    public $sort_order = 0;

    protected function rules()
    {
        return [
            "category_name" => "required|string|max:255",
            "uri" =>
                "required|string|max:255|unique:categories,uri," .
                $this->category["id"],
            "type" => "required|string",
            "display" => "required|boolean",
            "parent_id" => "nullable|exists:categories,id",
            "sort_order" => "nullable|integer|min:0",
        ];
    }

    protected $messages = [
        "category_name.required" => "Category name is required.",
        "uri.required" => "URL slug is required.",
        "uri.unique" => "This URL slug is already in use.",
        "type.required" => "Category type is required.",
        "parent_id.exists" => "Selected parent category does not exist.",
    ];

    public function mount($category)
    {
        // Handle different input types
        if ($category instanceof Categories) {
            // Route model binding passed a model instance
            $categoryModel = $category;
            $category = $category->toArray();
        } elseif (is_array($category)) {
            // Array was passed
            $categoryModel = Categories::find($category["id"]);
        } else {
            // String/integer ID was passed (most common case from route parameter)
            $categoryModel = Categories::find($category);
            $category = $categoryModel ? $categoryModel->toArray() : null;
        }

        // Ensure category was found
        if (!$category || !$categoryModel) {
            abort(404, 'Category not found');
        }

        $this->category = $category;
        $this->category_name = $category["category_name"];
        $this->uri = $category["uri"];
        // Convert enum to string value
        $this->type = $category["type"] instanceof \BackedEnum ? $category["type"]->value : $category["type"];
        $this->display = $category["display"];
        $this->parent_id = $category["parent_id"];
        $this->sort_order = $category["sort_order"] ?? 0;

        // Get content count (articles or announcements)
        $typeValue = $categoryModel->type instanceof \BackedEnum ? $categoryModel->type->value : $categoryModel->type;

        if ($typeValue === "knowledgebase") {
            // Only check if Knowledge Base extension is installed
            $kbExtension = \App\Models\Extension::where('extension_id', 'knowledgebase')
                ->where('enabled', true)
                ->first();

            if ($kbExtension && class_exists('\App\Extensions\Installed\Knowledgebase\Models\Article')) {
                $this->content_count = $categoryModel->knowledgebase()->count();
            }
        } elseif ($typeValue === "announcements") {
            $this->content_count = $categoryModel->announcements()->count();
        }
    }

    public function updatedCategoryName()
    {
        if ($this->auto_generate_uri) {
            $this->uri = Str::slug($this->category_name);
        }
    }

    public function updatedUri()
    {
        $this->uri = Str::slug($this->uri);
    }

    public function toggleAutoGenerateUri()
    {
        $this->auto_generate_uri = !$this->auto_generate_uri;

        if ($this->auto_generate_uri && $this->category_name) {
            $this->uri = Str::slug($this->category_name);
        }
    }

    public function save()
    {
        $this->validate();

        $categoryModel = Categories::findOrFail($this->category["id"]);

        // Validate parent relationship if parent is being changed
        if ($this->parent_id != $categoryModel->parent_id) {
            $validation = $categoryModel->canHaveParent($this->parent_id, 3);

            if (!$validation["valid"]) {
                session()->flash("error", $validation["error"]);
                return;
            }
        }

        $categoryModel->update([
            "category_name" => $this->category_name,
            "uri" => $this->uri,
            "type" => $this->type,
            "display" => $this->display,
            "parent_id" => $this->parent_id ?: null,
            "sort_order" => $this->sort_order,
        ]);

        session()->flash("success", "Category updated successfully!");

        return redirect()->route("settings.categories");
    }

    public function render()
    {
        $categoryModel = Categories::find($this->category["id"]);

        // Get all categories of the same type for parent dropdown (exclude self and descendants, max 2 levels)
        $availableParents = Categories::where("type", $this->type)
            ->where("id", "!=", $this->category["id"]) // Exclude self
            ->with("parent")
            ->get()
            ->filter(function ($category) use ($categoryModel) {
                // Exclude descendants of this category
                if (
                    $categoryModel &&
                    $category->ancestors()->contains("id", $categoryModel->id)
                ) {
                    return false;
                }
                // Only allow up to 2 levels deep
                return $category->getDepth() < 2;
            })
            ->sortBy("category_name");

        // Get available types
        $availableTypes = [
            "announcements" => "Announcements",
        ];

        // Only show Knowledge Base if extension is installed and enabled
        $knowledgebaseExtension = \App\Models\Extension::where('extension_id', 'knowledgebase')
            ->where('enabled', true)
            ->first();

        if ($knowledgebaseExtension) {
            $availableTypes = [
                "knowledgebase" => "Knowledge Base",
            ] + $availableTypes;
        }

        return view("livewire.settings.categories.edit", [
            "availableParents" => $availableParents,
            "availableTypes" => $availableTypes,
        ]);
    }
}
