<?php

namespace App\Livewire\Clients;

use App\Models\{User};

use Illuminate\Support\Facades\{Auth,DB};
use Illuminate\Support\{Str,Carbon,HtmlString};
use Illuminate\Database\Eloquent\Builder;

use Livewire\Component;
use Filament\Notifications\Notification;

use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Schemas\Concerns\InteractsWithSchemas;
use Filament\Schemas\Contracts\HasSchemas;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ViewColumn;
use Filament\Actions\Action;

class Assigned extends Component implements HasTable, HasSchemas, HasActions
{
    use InteractsWithActions;
    use InteractsWithSchemas;
    use InteractsWithTable;

    public function makeFilamentTranslatableContentDriver(): ?\Filament\Support\Contracts\TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        $query = User::query()
            ->select('users.*')
            ->where('account_manager', '=', Auth::user()->id)
            ->selectRaw('(SELECT COUNT(*) FROM sessions WHERE sessions.user_id = users.id AND sessions.user_id IS NOT NULL) as is_online');

        return $table
            ->query($query)
            ->heading('Assigned Customers')
            ->columns([
                ViewColumn::make('avatar')
                    ->label('')
                    ->view('filament.tables.columns.user-avatar')
                    ->toggleable()
                    ->extraAttributes(['class' => 'w-24']),

                TextColumn::make('name')
                    ->label(strtoupper(__('Name')))
                    ->searchable()
                    ->sortable()
                    ->toggleable()
                    ->extraAttributes(['class' => 'font-medium']),

                TextColumn::make('company')
                    ->label(strtoupper(__('Company')))
                    ->searchable()
                    ->sortable()
                    ->toggleable()
                    ->placeholder('—'),

                TextColumn::make('email')
                    ->label(strtoupper(__('Email Address')))
                    ->searchable()
                    ->sortable()
                    ->toggleable()
                    ->copyable()
                    ->copyMessage('Email copied!')
                    ->icon('heroicon-o-envelope'),

                TextColumn::make('account_manager_name')
                    ->label(strtoupper(__('Account Manager')))
                    ->getStateUsing(function ($record) {
                        if ($record->account_manager) {
                            $manager = User::find($record->account_manager);
                            return $manager ? $manager->name : '—';
                        }
                        return '—';
                    })
                    ->toggleable()
                    ->placeholder('—'),
            ])
            ->defaultSort('id', 'desc')
            ->actions([
                Action::make('view')
                    ->label('View')
                    ->button()
                    ->size('sm')
                    ->color('primary')
                    ->url(fn (User $record): string =>
                        $record->id != Auth::user()->id
                            ? route('view_client', ['clientId' => $record->id])
                            : route('settings.profile')
                    )
                    ->extraAttributes(['class' => '!bg-indigo-700 hover:!bg-indigo-500 !text-white'])
            ])
            ->emptyStateHeading(__("You don't have any assigned customers."))
            ->emptyStateIcon('heroicon-o-users')
            ->poll('5s')
            ->striped()
            ->paginated([10, 25, 50, 100]);
    }

    public function render()
    {
        return view('livewire.clients.assigned');
    }
}
