<?php

namespace App\Livewire\Attachments;

use App\Models\Attachment;
use App\Models\Tickets;
use App\Services\AttachmentService;
use Illuminate\Support\Facades\Auth;
use Livewire\Attributes\Computed;
use Livewire\Component;
use Livewire\WithFileUploads;
use Filament\Notifications\Notification;

class Upload extends Component
{
    use WithFileUploads;

    public $ticketId;
    public $responseId = null;
    public $files = [];
    public $uploadedAttachments = [];
    public $settings = [];
    public $uploading = false;

    protected $listeners = ['attachmentDeleted' => '$refresh'];

    public function mount($ticketId, $responseId = null)
    {
        $this->ticketId = $ticketId;
        $this->responseId = $responseId;

        // Load settings for this ticket's department
        $ticket = Tickets::find($ticketId);
        if ($ticket) {
            $this->settings = AttachmentService::getSettingsForDepartment($ticket->department_id);
        }

        // Load existing attachments
        $this->loadAttachments();
    }

    public function loadAttachments()
    {
        $query = Attachment::where('ticket_id', $this->ticketId);

        if ($this->responseId) {
            $query->where('response_id', $this->responseId);
        }

        $this->uploadedAttachments = $query->orderBy('created_at', 'desc')->get();
    }

    public function updatedFiles()
    {
        // Validate files on upload
        $this->validate([
            'files.*' => [
                'file',
                'max:' . ($this->settings['max_file_size_mb'] * 1024), // Convert MB to KB for validation
            ],
        ], [
            'files.*.max' => 'File size must not exceed ' . $this->settings['max_file_size_mb'] . 'MB.',
        ]);
    }

    public function upload()
    {
        if (empty($this->files)) {
            Notification::make()
                ->title('No files selected')
                ->warning()
                ->body('Please select at least one file to upload.')
                ->send();
            return;
        }

        $this->uploading = true;

        try {
            $uploadedCount = 0;
            $errors = [];

            foreach ($this->files as $file) {
                try {
                    AttachmentService::upload(
                        $file,
                        $this->ticketId,
                        Auth::id(),
                        $this->responseId,
                        false
                    );
                    $uploadedCount++;
                } catch (\Exception $e) {
                    $errors[] = $file->getClientOriginalName() . ': ' . $e->getMessage();
                }
            }

            // Clear files array
            $this->files = [];
            $this->uploading = false;

            // Reload attachments
            $this->loadAttachments();

            // Show notification
            if ($uploadedCount > 0) {
                Notification::make()
                    ->title('Files uploaded successfully')
                    ->success()
                    ->body("{$uploadedCount} file(s) uploaded successfully.")
                    ->send();
            }

            if (!empty($errors)) {
                Notification::make()
                    ->title('Some files failed to upload')
                    ->warning()
                    ->body(implode("\n", $errors))
                    ->send();
            }

            // Dispatch event for parent components
            $this->dispatch('attachmentsUploaded', ticketId: $this->ticketId);

        } catch (\Exception $e) {
            $this->uploading = false;

            Notification::make()
                ->title('Upload failed')
                ->danger()
                ->body($e->getMessage())
                ->send();
        }
    }

    public function deleteAttachment($attachmentId)
    {
        try {
            $attachment = Attachment::findOrFail($attachmentId);

            // Check authorization
            if (!AttachmentService::canDownload($attachment, Auth::user())) {
                Notification::make()
                    ->title('Unauthorized')
                    ->danger()
                    ->body('You do not have permission to delete this attachment.')
                    ->send();
                return;
            }

            // Delete attachment
            AttachmentService::delete($attachment);

            // Reload attachments
            $this->loadAttachments();

            Notification::make()
                ->title('Attachment deleted')
                ->success()
                ->body('The attachment has been deleted successfully.')
                ->send();

            // Dispatch event
            $this->dispatch('attachmentDeleted', attachmentId: $attachmentId);

        } catch (\Exception $e) {
            Notification::make()
                ->title('Deletion failed')
                ->danger()
                ->body($e->getMessage())
                ->send();
        }
    }

    #[Computed]
    public function maxFiles()
    {
        return $this->settings['max_files_per_ticket'] ?? 10;
    }

    #[Computed]
    public function canUpload()
    {
        return $this->uploadedAttachments->count() < $this->maxFiles;
    }

    public function getAcceptedFileTypes()
    {
        $allowed = $this->settings['allowed_file_types'] ?? [];

        if (empty($allowed)) {
            return '*';
        }

        // Convert extensions to accept attribute format
        $extensions = array_filter($allowed, fn($type) => !str_contains($type, '/'));

        return !empty($extensions) ? '.' . implode(',.', $extensions) : '*';
    }

    public function render()
    {
        return view('livewire.attachments.upload');
    }
}
