<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;

class LiveChatCors
{
    public function handle(Request $request, Closure $next)
    {
        // Only apply CORS to specific routes
        if (!$request->is('broadcasting/auth') && !$request->is('extensions/live-chat/api/chat/*')) {
            return $next($request);
        }

        // Check if LiveChat extension classes exist
        if (!class_exists('\App\Extensions\Installed\LiveChat\Services\WidgetCors') ||
            !class_exists('\App\Extensions\Installed\LiveChat\Models\ChatWidgetSetting')) {
            // Extension not installed, allow request to proceed
            return $next($request);
        }

        try {
            $origin = $request->headers->get('Origin');

            // If no origin header, allow the request (same-origin requests)
            if (!$origin) {
                return $next($request);
            }

            // Check if request is from the app itself (same origin as APP_URL)
            $appOrigin = rtrim(config('app.url'), '/');
            $currentOrigin = rtrim($request->getSchemeAndHttpHost(), '/');
            $normalizedOrigin = rtrim($origin, '/');

            $widgetCorsClass = '\App\Extensions\Installed\LiveChat\Services\WidgetCors';

            if (
                ($appOrigin && strcasecmp($normalizedOrigin, $appOrigin) === 0) ||
                ($currentOrigin && strcasecmp($normalizedOrigin, $currentOrigin) === 0)
            ) {
                // Request from the app itself (any configured or current host) - allow without widget checks
                $response = $next($request);
                return $widgetCorsClass::addCorsHeaders($response, $origin);
            }
        } catch (\Exception $e) {
            \Log::error('LiveChat CORS middleware error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'url' => $request->fullUrl(),
                'origin' => $request->headers->get('Origin'),
            ]);
            // On error, allow the request to proceed
            return $next($request);
        }

        // For external widget requests, check widget key and settings
        $widgetKey = $this->resolveWidgetKey($request);
        $widget = null;

        if ($widgetKey) {
            try {
                $chatWidgetSettingClass = '\App\Extensions\Installed\LiveChat\Models\ChatWidgetSetting';
                $widget = $chatWidgetSettingClass::where('embed_key', $widgetKey)
                    ->where('enabled', true)
                    ->first();
            } catch (\Exception $e) {
                // Database connection might not be available (local dev environment)
                // Continue anyway for development purposes
                \Log::warning('LiveChat CORS: Could not query widget settings', [
                    'error' => $e->getMessage()
                ]);
            }
        }

        $widgetCorsClass = '\App\Extensions\Installed\LiveChat\Services\WidgetCors';
        $allowedOrigin = $widgetCorsClass::resolveAllowedOrigin($request, $widget);
        if (!$allowedOrigin) {
            return $widgetCorsClass::forbiddenOriginResponse();
        }

        // Consider it a widget request if:
        // 1. Widget was found in DB, OR
        // 2. Widget key was provided in request (even if not found), OR
        // 3. OPTIONS preflight expects widget key header
        $isWidgetRequest = (bool) $widget || (bool) $widgetKey || $this->requestExpectsWidgetKey($request);

        if (!$isWidgetRequest) {
            // Not a widget request and not from app origin - reject
            return $widgetCorsClass::forbiddenOriginResponse();
        }

        if ($request->isMethod('options')) {
            return $widgetCorsClass::addCorsHeaders(response('', 204), $allowedOrigin);
        }

        $response = $next($request);

        return $widgetCorsClass::addCorsHeaders($response, $allowedOrigin);
    }

    protected function resolveWidgetKey(Request $request): ?string
    {
        return $request->header('X-Widget-Key')
            ?? $request->query('widget_key')
            ?? $request->input('widget_key')
            ?? $request->query('key');
    }

    protected function requestExpectsWidgetKey(Request $request): bool
    {
        if (!$request->isMethod('options')) {
            return false;
        }

        $requestedHeaders = array_filter(array_map(function ($header) {
            return strtolower(trim($header));
        }, explode(',', $request->header('Access-Control-Request-Headers', ''))));

        return in_array('x-widget-key', $requestedHeaders, true);
    }
}