<?php

namespace App\Http\Controllers\Api\Two;

use App\Http\Controllers\{Controller, TicketsController};
use App\Http\Controllers\Api\Two\{ApiController};
use App\Models\{User, Tickets, Departments};
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\{Hash, Auth, Http};

class ApiTickets extends Controller
{
    /**
     * Get all tickets for a user.
     * This method retrieves all tickets associated with the authenticated user.
     */
    public function getTickets(Request $request)
    {
        $user = User::select('*')->where('id', $request->id)->first();

        if (!$user) {
            return response()->json(['error' => 'User not found'], 401);
        }

        $tickets = Tickets::where('user_id', $user['id'])
        ->leftJoin('departments', 'tickets.department_id', '=', 'departments.id')
        ->select('tickets.*', 'departments.*')
        ->selectRaw('departments.id as department_unique_id, tickets.created_at as ticket_created, departments.created_at as department_created, tickets.id as id')
        ->get();

        if (!$tickets) {
            return response()->json(['error' => 'No tickets found for this user.'], 404);
        } else {
            return $tickets;
        }
    }

    /**
     * Get the ticket for a user.
     */
    public function getTicketInfo(Request $request)
    {
        $connection = ApiController::connection($request->header("Authorization"));
        if ($connection->content() != '{"result":"true"}') {
            return $connection;
        }

        $tickets = Tickets::where('tickets.id', '=', $request->ticket_id)
            ->leftJoin('users', 'tickets.user_id', '=', 'users.id')
            ->leftJoin('departments', 'tickets.department_id', '=', 'departments.id')
            ->select('tickets.*', 'departments.*', 'users.name', 'users.email','users.company')
            ->selectRaw('users.company as company_name, users.name as client_name, departments.id as department_unique_id, tickets.created_at as ticket_created')
            ->first();
        
        if (!$tickets) {
            return response()->json(['error' => 'No tickets found with this id.'], 404);
        } else {
            return response()->json(['tickets' => $tickets], 200);
        }
    }

     /**
     * Get the ticket for a user.
     */
    public function getTicket(Request $request)
    {
        $connection = ApiController::connection($request->header("Authorization"));
        if ($connection->content() != '{"result":"true"}') {
            return $connection;
        }
        $user = User::select('*')->where('id', $request->id)->first();
        
        if (!$user) {
            return response()->json(['error' => 'User not found.'], 401);
        }

        if (!$request->ticket_id) {
            return response()->json(['error' => 'No ticket id provided.'], 401);
        }

        $tickets = Tickets::where('tickets.user_id', '=', $request->id)
            ->where('tickets.id', '=', $request->ticket_id)
            ->leftJoin('users', 'tickets.user_id', '=', 'users.id')
            ->leftJoin('departments', 'tickets.department_id', '=', 'departments.id')
            ->select('tickets.*', 'departments.*', 'users.name', 'users.email','users.company')
            ->selectRaw('users.company as company_name, users.name as client_name, departments.id as department_unique_id, tickets.created_at as ticket_created')
            ->first();
        
        if (!$tickets) {
            return response()->json(['error' => 'No tickets found for this user.'], 404);
        } else {
            return response()->json(['tickets' => $tickets], 200);
        }
    }

        /**
         *  Creates a new ticket.
         * user_id      = The Customer's Ticaga ID [REQUIRED]
         * subject      = Subject of the ticket [REQUIRED]
         * message      = Content of the ticket [REQUIRED]
         * status       = Status of the ticket, will default to open
         * priority     = Priority of the ticket, will default to low
         * cc           = optional
         * assigned     = optional
         * public_hash  = optional
         * public_name  = optional
         * public_email = optional
         * ip_address   = IP Address of the account [REQUIRED]
         */
        public function store(Request $request)
        {         
            $connection = ApiController::connection($request->header("Authorization"));
            if ($connection->content() != '{"result":"true"}') {
                return $connection;
            }
            
            if (!$request->department_slug) {
                $department_id = $request->department_id;
            }else{
                $department_id = Departments::select('id')->where('slug', '=', $request->department_slug)->first()->getOriginal();
            }
            
            $token = bin2hex(random_bytes(16));
            $public_hash = hash_hmac('sha256', $token, config('app.key'));

            if($request->customer_id == '0' || !$request->customer_id) {

                $ticket = Tickets::create([
                    'subject' => $request->subject ?? 'No Subject',
                    'message' => $request->message,
                    'status' => 'open',
                    'priority' => Str::lower($request->priority) ?? 'low',
                    'cc' => $request->cc ?? '',
                    'assigned' => '0',
                    'department_id' => $department_id['id'] ?? $request->department_id,
                    'public_hash' => $public_hash,
                    'public_name' => $request->public_name ?? null,
                    'public_email' => $request->public_email ?? null,
                    'rating' => '0',
                    'ip_address' => $request->ip_address ?? app('request')->ip(),
                    'organize' => Str::lower($request->organize) ?: 'ticket',
                    'date_closed' => null,
                    'updated_by_client_at' => date("Y-m-d H:i:s"),
                    'created_at' => date("Y-m-d H:i:s"),
                    'updated_at' => null,
                    'first_employee_reply', null,
                ]);
            } else {
                
                $account_manager = User::select('account_manager')->where('id', $request->customer_id)->first();
            
                $ticket = Tickets::create([
                    'user_id' => $request->customer_id ?? '0',
                    'subject' => $request->subject ?? 'No Subject',
                    'message' => $request->message,
                    'status' => 'open',
                    'priority' => Str::lower($request->priority) ?? 'low',
                    'cc' => $request->cc ?? '',
                    'assigned' => $account_manager->account_manager ?? '0',
                    'department_id' => $department_id['id'] ?? $request->department_id,
                    'public_hash' => $public_hash,
                    'public_name' => $request->public_name ?? null,
                    'public_email' => $request->public_email ?? null,
                    'rating' => '0',
                    'ip_address' => $request->ip_address ?? app('request')->ip(),
                    'organize' => Str::lower($request->organize) ?: 'ticket',
                    'date_closed' => null,
                    'updated_by_client_at' => date("Y-m-d H:i:s"),
                    'created_at' => date("Y-m-d H:i:s"),
                    'updated_at' => null,
                    'first_employee_reply', null,
                ]);
            }

            if (!$ticket) {
                return response()->json(['error' => "Failed to create the ticket."], 404);
            } else {
                return response()->json(['id' => $ticket], 201);
            }
        }

        /**
         * Close a ticket.
         * id = The ID of the user who owns the ticket [REQUIRED]
         * ticket_id = The ID of the ticket to close [REQUIRED]
         */
        public function close(Request $request)
        {
            $connection = ApiController::connection($request->header("Authorization"));
            if ($connection->content() != '{"result":"true"}') {
                return $connection;
            }

            if($request->customer_id != '0'){
                $user = User::select('*')->where('id', $request->customer_id)->first();
                
                if (!$user) {
                    return response()->json(['error' => 'User not found.'], 401);
                }
            }

            if (!$request->ticket_id) {
                return response()->json(['error' => 'No ticket id provided.'], 401);
            }

            $tickets = Tickets::where('tickets.user_id', '=', $request->customer_id)
                ->where('tickets.id', '=', $request->ticket_id)
                ->leftJoin('users', 'tickets.user_id', '=', 'users.id')
                ->select('tickets.*','users.name', 'users.email','users.company')
                ->selectRaw('tickets.id as ticket_number')
                ->first();
            
            if (!$tickets) {
                return response()->json(['error' => 'No tickets found for this user.'], 404);
            } else {
                $ticket = TicketsController::destroy($tickets);
                return response()->json(['id' => $tickets->ticket_number], 200);
            }
        }
}