<?php

use App\Extensions\Installed\Importer\Models\ImportJob;
use App\Extensions\Installed\Importer\Services\CsvImporter;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use function Livewire\Volt\{state, mount};

state([
    'job',
    'headers' => [],
    'mapping' => [],
    'requiredFields' => [],
    'optionalFields' => [],
    'importTypeConfig' => [],
    'previewRows' => [],
]);

mount(function (ImportJob $job) {
    $this->job = $job;

    // Get CSV headers and preview data
    $csvImporter = new CsvImporter($job->file_path, $job->type);
    $this->headers = $csvImporter->getHeaders();

    $preview = $csvImporter->preview(5);
    $this->previewRows = collect($preview)->map(function (array $row) {
        $padded = array_pad($row, count($this->headers), null);
        return array_combine($this->headers, $padded);
    })->all();

    // Default field definitions (mirrors upload step)
    $defaultConfig = [
        'customers' => [
            'required_fields' => ['name', 'email'],
            'optional_fields' => [
                'company',
                'phone',
                'phone_number',
                'password',
                'billing_id',
                'billing_system',
                'account_manager',
                'force_password_reset',
            ],
        ],
        'departments' => [
            'required_fields' => ['department_name'],
            'optional_fields' => [
                'department_description',
                'slug',
                'department_email',
                'allows_high_priority',
                'cc_enabled',
                'is_public',
                'is_disabled',
                'soft_deleted',
                'created_at',
                'updated_at',
            ],
        ],
        'tickets' => [
            'required_fields' => ['subject'],
            'optional_fields' => [
                'user_id',
                'user_email',
                'public_name',
                'public_email',
                'message',
                'status',
                'priority',
                'department_id',
                'department_name',
                'assigned',
                'ip_address',
                'organize',
                'created_at',
                'updated_at',
            ],
        ],
        'responses' => [
            'required_fields' => ['ticket_id', 'content'],
            'optional_fields' => [
                'ticket_number',
                'user_id',
                'employee_response',
                'is_employee',
                'is_note',
                'organize',
                'ip_address',
                'created_at',
                'updated_at',
            ],
        ],
    ];

    $configured = config("extensions.importer.import_types.{$job->type}", []);
    $mergedConfig = array_replace_recursive($defaultConfig[$job->type] ?? [], $configured);

    $this->importTypeConfig = $mergedConfig;
    $this->requiredFields = Arr::wrap($mergedConfig['required_fields'] ?? []);
    $this->optionalFields = Arr::wrap($mergedConfig['optional_fields'] ?? []);

    if ($job->type === 'responses') {
        $this->optionalFields = array_values(array_unique(array_merge(
            $this->optionalFields,
            ['ticket_number', 'employee_response', 'organize', 'ip_address', 'updated_at']
        )));
    }

    // Initialize mapping with automatic guesses
    $this->mapping = $this->autoMapFields();
});

$autoMapFields = function () {
    $mapping = [];
    $allFields = array_merge($this->requiredFields, $this->optionalFields);

    foreach ($this->headers as $header) {
        $normalized = strtolower(str_replace([' ', '_', '-'], '', $header));

        foreach ($allFields as $field) {
            $normalizedField = strtolower(str_replace([' ', '_', '-'], '', $field));

            if ($normalized === $normalizedField ||
                str_contains($normalized, $normalizedField) ||
                str_contains($normalizedField, $normalized)) {
                $mapping[$header] = $field;
                break;
            }
        }
    }

    return $mapping;
};

if (!function_exists('ticaga_format_import_field_label')) {
    function ticaga_format_import_field_label(string $field): string
    {
        return match ($field) {
            'user_id' => 'Customer ID',
            'user_email' => 'Customer Email',
            'public_email' => 'Public Email',
            'public_name' => 'Public Name',
            'department_id' => 'Department ID',
            'department_name' => 'Department Name',
            'ip_address' => 'IP Address',
            'organize' => 'Organize (Type)',
            'ticket_number' => 'Ticket Number',
            'employee_response' => 'Employee Response',
            'is_public' => 'Is Public',
            'is_disabled' => 'Is Disabled',
            'soft_deleted' => 'Soft Deleted',
            'allows_high_priority' => 'Allows High Priority',
            'cc_enabled' => 'CC Enabled',
            'createdat' => 'Create Date',
            'createdate' => 'Create Date',
            'created_at' => 'Create Date',
            'updatedat' => 'Update Date',
            'updatedate' => 'Update Date',
            'updated_at' => 'Update Date',
            default => ucwords(str_replace('_', ' ', $field)),
        };
    }
}

if (!function_exists('ticaga_normalize_import_field')) {
    function ticaga_normalize_import_field(?string $field): ?string
    {
        if ($field === null) {
            return null;
        }

        $normalized = strtolower(trim($field));
        $normalized = str_replace(['*', ':'], '', $normalized);

        $condensed = str_replace([' ', '-', '_'], '', $normalized);

        $lookup = [
            'customeremail' => 'user_email',
            'customerid' => 'user_id',
            'publicemail' => 'public_email',
            'publicname' => 'public_name',
            'departmentname' => 'department_name',
            'departmentid' => 'department_id',
            'createdat' => 'created_at',
            'createdate' => 'created_at',
            'updatedat' => 'updated_at',
            'updatedate' => 'updated_at',
            'ticketnumber' => 'ticket_number',
            'employeeresponse' => 'employee_response',
            'isemployeeresponse' => 'is_employee',
            'organize' => 'organize',
            'ipaddress' => 'ip_address',
        ];

        if (isset($lookup[$condensed])) {
            return $lookup[$condensed];
        }

        return str_replace([' ', '-'], '_', $normalized);
    }
}

$saveMapping = function () {
    $filteredMapping = array_filter(
        $this->mapping,
        fn($value) => $value !== null && $value !== ''
    );

    // Validate that all required fields are mapped
    $normalizedMapping = collect($filteredMapping)
        ->map(fn($field) => ticaga_normalize_import_field($field))
        ->filter();

    $normalizedHeaders = collect(array_keys($filteredMapping))
        ->map(fn($header) => ticaga_normalize_import_field($header))
        ->filter();

    $mappedFields = $normalizedMapping->values()->all();
    $missingFields = [];

    if ($this->job->type === 'tickets') {
        $subjectMapped = $normalizedMapping->contains('subject') || $normalizedHeaders->contains('subject');
        if (!$subjectMapped) {
            $missingFields[] = 'Subject';
        }

        $hasUserIdentifier = $normalizedMapping->contains(fn($value) => in_array($value, ['user_id', 'user_email', 'public_email'], true))
            || $normalizedHeaders->contains(fn($value) => in_array($value, ['user_id', 'user_email', 'public_email'], true));

        if (!$hasUserIdentifier) {
            $missingFields[] = 'Customer ID / Email / Public Email';
        }

        $hasDepartment = $normalizedMapping->contains(fn($value) => in_array($value, ['department_id', 'department_name'], true))
            || $normalizedHeaders->contains(fn($value) => in_array($value, ['department_id', 'department_name'], true));

        if (!$hasDepartment) {
            $missingFields[] = 'Department ID / Name';
        }
    } elseif ($this->job->type === 'responses') {
        $ticketMapped = $normalizedMapping->contains(fn($value) => in_array($value, ['ticket_id', 'ticket_number'], true))
            || $normalizedHeaders->contains(fn($value) => in_array($value, ['ticket_id', 'ticket_number'], true));

        if (!$ticketMapped) {
            $missingFields[] = 'Ticket ID / Ticket Number';
        }

        $contentMapped = $normalizedMapping->contains('content') || $normalizedHeaders->contains('content');

        if (!$contentMapped) {
            $missingFields[] = 'Content';
        }
    } else {
        foreach ($this->requiredFields as $field) {
            $normalizedField = ticaga_normalize_import_field($field);
            if (!$normalizedMapping->contains($normalizedField)) {
                $missingFields[] = ticaga_format_import_field_label($normalizedField ?? $field);
            }
        }
    }

    if (!empty($missingFields)) {
        session()->flash('error', 'Please map all required fields: ' . implode(', ', $missingFields));
        return;
    }

    // Update job with mapping
    $this->job->update(['field_mapping' => $filteredMapping]);

    return redirect()->route('extensions.importer.csv.process', ['jobId' => $this->job->id]);
};

?>

@include('extension.importer::partials.navigation')

    @volt('extensions.importer.csv.mapping')
    <div class="space-y-6 pb-16">
        <!-- Breadcrumb -->
        <nav class="flex text-sm text-gray-500 dark:text-gray-400" aria-label="Breadcrumb">
            <ol class="inline-flex items-center space-x-1 md:space-x-3">
                <li class="inline-flex items-center">
                    <a href="{{ route('extensions.importer.index') }}"
                       class="inline-flex items-center gap-1 text-gray-600 hover:text-indigo-500 dark:text-gray-300 dark:hover:text-indigo-400 transition">
                        <x-heroicon-o-arrow-uturn-left class="w-4 h-4" />
                        Importer
                    </a>
                </li>
                <li>
                    <div class="flex items-center">
                        <x-heroicon-s-chevron-right class="w-4 h-4 text-gray-400" />
                        <span class="ml-1 text-gray-500 dark:text-gray-400">Field Mapping</span>
                    </div>
                </li>
            </ol>
        </nav>

        <!-- Header Card -->
        <div class="bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-xl shadow-sm p-6">
            <div class="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                <div>
                    <h1 class="text-2xl font-semibold text-gray-900 dark:text-white">Map CSV Columns</h1>
                    <p class="mt-2 text-sm text-gray-600 dark:text-gray-400 max-w-2xl">
                        Step 2 of 3. Match each column from your CSV with the Ticaga field it should populate.
                        Required fields are marked with an asterisk (*).
                    </p>
                </div>
                <div class="flex items-center gap-3">
                    <span class="inline-flex items-center rounded-full border border-indigo-200/70 dark:border-indigo-500/40 bg-indigo-50 dark:bg-indigo-500/10 px-3 py-1 text-sm font-medium text-indigo-600 dark:text-indigo-300">
                        {{ ucfirst($job->type) }} import
                    </span>
                    <span class="text-sm text-gray-500 dark:text-gray-400">Step 2 / 3</span>
                </div>
            </div>

            <div class="mt-4 h-2 w-full rounded-full bg-gray-200 dark:bg-gray-700">
                <div class="h-2 rounded-full bg-indigo-500" style="width: 66%"></div>
            </div>
        </div>

        @if (session('error'))
            <div class="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 text-red-800 dark:text-red-200 rounded-xl px-4 py-3">
                {{ session('error') }}
            </div>
        @endif

        <div class="grid gap-6 lg:grid-cols-3 lg:items-start">
            <!-- Mapping Column -->
            <div class="lg:col-span-2 space-y-6">
                <div class="bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-xl shadow-sm p-6">
                    <div class="flex items-center justify-between flex-wrap gap-3 mb-4">
                        <div>
                            <h2 class="text-lg font-semibold text-gray-900 dark:text-white">Column Mappings</h2>
                            <p class="text-sm text-gray-600 dark:text-gray-400">
                                We pre-selected matches where possible. Review the suggestions and adjust any fields that need attention.
                            </p>
                        </div>
                        <div class="flex items-center gap-2 text-xs text-gray-500 dark:text-gray-400">
                            <span class="inline-flex items-center gap-1 rounded-full bg-indigo-50 px-2.5 py-1 dark:bg-indigo-500/10 text-indigo-600 dark:text-indigo-300">
                                <span class="h-2 w-2 rounded-full bg-indigo-500 dark:bg-indigo-300"></span>
                                Auto mapped
                            </span>
                            <span class="inline-flex items-center gap-1 rounded-full bg-red-50 px-2.5 py-1 dark:bg-red-500/10 text-red-600 dark:text-red-300">
                                <span class="h-2 w-2 rounded-full bg-red-500 dark:bg-red-300"></span>
                                Needs mapping
                            </span>
                        </div>
                    </div>

                    <div class="space-y-3">
                        @foreach($headers as $header)
                            @php
                                $currentMapping = $mapping[$header] ?? '';
                                $sampleValue = collect($previewRows)->pluck($header)->filter()->first();
                                $mappingLabel = $currentMapping ? ticaga_format_import_field_label($currentMapping) : null;
                            @endphp
                            <div class="rounded-xl border border-gray-200 bg-white px-4 py-3 shadow-sm dark:border-gray-700 dark:bg-gray-800">
                                <div class="flex flex-col gap-3 md:flex-row md:items-center md:justify-between">
                                    <div class="space-y-1">
                                        <div class="flex items-center gap-2">
                                            <span class="text-sm font-medium text-gray-900 dark:text-white">{{ $header }}</span>
                                            @if($mappingLabel)
                                                <span class="inline-flex items-center gap-1 rounded-full bg-indigo-100 px-2 py-0.5 text-xs font-medium text-indigo-600 dark:bg-indigo-500/20 dark:text-indigo-200">
                                                    {{ $mappingLabel }}
                                                    @if(in_array($currentMapping, $requiredFields))
                                                        *
                                                    @endif
                                                </span>
                                            @endif
                                        </div>
                                        <p class="text-xs text-gray-500 dark:text-gray-300">
                                            Sample value:
                                            <span class="font-medium text-gray-700 dark:text-gray-100">
                                                {{ $sampleValue !== null && $sampleValue !== '' ? Str::limit($sampleValue, 40) : '—' }}
                                            </span>
                                        </p>
                                    </div>
                                    <div class="md:w-72">
                                        <select wire:model="mapping.{{ $header }}"
                                                class="w-full rounded-lg border border-gray-300 bg-white text-gray-900 shadow-sm transition focus:border-indigo-500 focus:ring-indigo-500 dark:border-gray-600 dark:bg-gray-900 dark:text-gray-100">
                                            <option value="">-- Skip this column --</option>

                                            @if(!empty($requiredFields))
                                                <optgroup label="Required Fields">
                                                    @foreach($requiredFields as $field)
                                                        <option value="{{ $field }}">{{ ticaga_format_import_field_label($field) }} *</option>
                                                    @endforeach
                                                </optgroup>
                                            @endif

                        @if(!empty($optionalFields))
                                                <optgroup label="Optional Fields">
                                                    @foreach($optionalFields as $field)
                                                        <option value="{{ $field }}">{{ ticaga_format_import_field_label($field) }}</option>
                                                    @endforeach
                                                </optgroup>
                                            @endif
                                        </select>
                                    </div>
                                </div>
                            </div>
                        @endforeach
                    </div>

                    @php
                        $filteredMapping = array_filter(
                            $mapping,
                            fn($value) => $value !== null && $value !== ''
                        );
                        $mappedFields = array_values($filteredMapping);
                        $normalizedMapping = collect($filteredMapping)
                            ->map(fn($field) => ticaga_normalize_import_field($field))
                            ->filter();

                        $normalizedHeaders = collect(array_keys($filteredMapping))
                            ->map(fn($header) => ticaga_normalize_import_field($header))
                            ->filter();

                        $mappedFields = $normalizedMapping->values()->all();

                        $requiredStatuses = collect($requiredFields)->map(function ($field) use ($normalizedMapping) {
                            $normalizedField = ticaga_normalize_import_field($field);

                            return [
                                'label' => ticaga_format_import_field_label($normalizedField ?? $field),
                                'mapped' => $normalizedField ? $normalizedMapping->contains($normalizedField) : false,
                            ];
                        })->values();

                        if ($job->type === 'tickets') {
                            $subjectMapped = $normalizedMapping->contains('subject') || $normalizedHeaders->contains('subject');
                            $customerMapped = $normalizedMapping->contains(fn($value) => in_array($value, ['user_id', 'user_email', 'public_email'], true))
                                || $normalizedHeaders->contains(fn($value) => in_array($value, ['user_id', 'user_email', 'public_email'], true));
                            $departmentMapped = $normalizedMapping->contains(fn($value) => in_array($value, ['department_id', 'department_name'], true))
                                || $normalizedHeaders->contains(fn($value) => in_array($value, ['department_id', 'department_name'], true));

                            $requiredStatuses = collect([
                                [
                                    'label' => 'Subject',
                                    'mapped' => $subjectMapped,
                                ],
                                [
                                    'label' => 'Customer ID / Email / Public Email',
                                    'mapped' => $customerMapped,
                                ],
                                [
                                    'label' => 'Department ID / Name',
                                    'mapped' => $departmentMapped,
                                ],
                            ]);
                        } elseif ($job->type === 'responses') {
                            $ticketMapped = $normalizedMapping->contains(fn($value) => in_array($value, ['ticket_id', 'ticket_number'], true))
                                || $normalizedHeaders->contains(fn($value) => in_array($value, ['ticket_id', 'ticket_number'], true));
                            $contentMapped = $normalizedMapping->contains('content') || $normalizedHeaders->contains('content');

                            $requiredStatuses = collect([
                                [
                                    'label' => 'Ticket ID / Ticket Number',
                                    'mapped' => $ticketMapped,
                                ],
                                [
                                    'label' => 'Content',
                                    'mapped' => $contentMapped,
                                ],
                            ]);
                        }
                    @endphp

                    <div class="mt-6 rounded-xl border border-blue-200 dark:border-blue-800 bg-blue-50 dark:bg-blue-900/20 p-4">
                        <h3 class="text-sm font-semibold text-blue-900 dark:text-blue-200 mb-3">Required field status</h3>
                        <div class="grid gap-2 sm:grid-cols-2">
                            @foreach($requiredStatuses as $status)
                                <div class="flex items-center gap-2 rounded-lg bg-white/70 px-3 py-2 text-sm shadow-sm dark:bg-blue-900/40">
                                    @if($status['mapped'])
                                        <x-heroicon-s-check-circle class="h-5 w-5 text-emerald-500" />
                                        <span class="text-emerald-700 dark:text-emerald-300">{{ $status['label'] }} mapped</span>
                                    @else
                                        <x-heroicon-s-x-circle class="h-5 w-5 text-red-500" />
                                        <span class="text-red-700 dark:text-red-300">{{ $status['label'] }} missing</span>
                                    @endif
                                </div>
                            @endforeach
                        </div>
                    </div>

                    <div class="mt-6 flex items-center justify-between gap-3">
                        <a href="{{ route('extensions.importer.csv.upload') }}"
                           class="inline-flex items-center justify-center gap-2 rounded-xl border border-gray-300 px-4 py-3 text-sm font-medium text-gray-700 transition hover:bg-gray-50 dark:border-gray-600 dark:text-gray-200 dark:hover:bg-gray-800">
                            <x-heroicon-s-arrow-left class="h-5 w-5" />
                            Back
                        </a>
                        <button wire:click="saveMapping"
                                class="inline-flex items-center justify-center gap-2 rounded-xl bg-indigo-600 px-5 py-3 text-sm font-semibold text-white shadow-sm transition hover:bg-indigo-700">
                            Continue to Preview
                            <x-heroicon-s-arrow-right class="h-5 w-5" />
                        </button>
                    </div>
                </div>
            </div>

            <!-- Sidebar -->
            <div class="space-y-6">
                <div class="rounded-xl border border-gray-200 bg-white p-6 shadow-sm dark:border-gray-700 dark:bg-gray-900">
                    <h3 class="text-lg font-semibold text-gray-900 dark:text-white">Data Preview</h3>
                    <p class="mt-2 text-sm text-gray-600 dark:text-gray-400">
                        A quick look at the first rows in your CSV. Use it to confirm the values align with the selected fields.
                    </p>
                    <div class="mt-4 space-y-4 text-sm">
                        @forelse($previewRows as $row)
                            <div class="rounded-lg border border-gray-200 bg-gray-50 p-4 dark:border-gray-700 dark:bg-gray-800/60">
                                <h4 class="text-xs font-semibold uppercase tracking-wide text-gray-500 dark:text-gray-400 mb-2">
                                    Row {{ $loop->iteration }}
                                </h4>
                                <dl class="space-y-1">
                                    @foreach($headers as $header)
                                        <div class="flex justify-between gap-3">
                                            <dt class="text-gray-500 dark:text-gray-400">{{ $header }}:</dt>
                                            <dd class="text-gray-800 dark:text-gray-200 font-medium text-right">
                                                {{ Str::limit($row[$header] ?? '', 40) ?: '—' }}
                                            </dd>
                                        </div>
                                    @endforeach
                                </dl>
                            </div>
                        @empty
                            <p class="text-sm text-gray-500 dark:text-gray-400">No preview data available.</p>
                        @endforelse
                    </div>
                </div>

                <div class="rounded-xl border border-yellow-200 bg-yellow-50 p-6 shadow-sm dark:border-yellow-800 dark:bg-yellow-900/20">
                    <div class="flex items-start gap-3 text-sm text-yellow-800 dark:text-yellow-200">
                        <x-heroicon-s-information-circle class="h-6 w-6 text-yellow-500" />
                        <div>
                            <h4 class="font-semibold mb-2">Mapping tips</h4>
                            <ul class="space-y-1">
                                <li>Auto suggestions come from matching column names.</li>
                                <li>You can map multiple CSV columns to the same field; the last value wins.</li>
                                <li>Columns left unmapped are ignored during import.</li>
                                <li>All required fields must be mapped before continuing.</li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    @endvolt
