<?php

namespace App\Extensions\Installed\Importer\Services\DatabaseAdapters;

/**
 * Contract for database import adapters.
 *
 * Each adapter encapsulates the SQL/schema differences for a specific helpdesk
 * or billing system (SupportPal, WHMCS, Blesta, ClientExec, etc.).
 */
interface DatabaseImportAdapter
{
    /**
     * Unique identifier for this adapter (e.g., supportpal, whmcs).
     */
    public static function id(): string;

    /**
     * Human readable label (SupportPal, WHMCS, etc.).
     */
    public static function label(): string;

    /**
     * Return associative array describing system specific defaults.
     *
     * @return array{
     *     chunk_size?: int,
     *     options?: array<string,mixed>
     * }
     */
    public static function defaults(): array;

    /**
     * Whether this adapter is production ready.
     */
    public static function isPreview(): bool;

    /**
     * Initialise the adapter with database configuration and options.
     *
     * @param array $dbConfig Connection config (host, port, database, username, password, prefix)
     * @param array $options  Import job options
     */
    public function boot(array $dbConfig, array $options = []): void;

    /**
     * Return ordered list of stages supported by this adapter.
     *
     * e.g. ['departments','categories','users','operators','tickets','responses','articles']
     *
     * @return array<int,string>
     */
    public function stages(): array;

    /**
     * Total rows for the given stage.
     */
    public function count(string $stage): int;

    /**
     * Fetch a chunk of rows for the provided stage.
     *
     * Implementations should return an array of associative arrays matching the
     * field expectations of DatabaseImporter.
     *
     * @return array<int,array<string,mixed>>
     */
    public function fetch(string $stage, int $offset, int $limit): array;

    /**
     * Map source status identifier to Ticaga status string.
     */
    public function mapStatus(?int $statusId): string;

    /**
     * Map source priority identifier to Ticaga priority string.
     */
    public function mapPriority(?int $priorityId): string;

    /**
     * Determine if the provided source user id belongs to an operator/staff account.
     */
    public function isOperatorUser(int $userId): bool;

    /**
     * Retrieve a source user row by id (for diagnostics).
     *
     * @return array<string,mixed>|null
     */
    public function findUser(int $userId): ?array;

    /**
     * Retrieve underlying PDO connection if direct queries are required.
     */
    public function connection(): \PDO;

    /**
     * Prefix used for source tables (if any).
     */
    public function prefix(): string;

    /**
     * Determine if a table exists in the source database.
     */
    public function tableExists(string $table): bool;
}
