<?php

namespace App\Extensions\Installed\Importer\Services\DatabaseAdapters;

use Illuminate\Support\Collection;

class DatabaseAdapterRegistry
{
    /**
     * @var array<string,class-string<DatabaseImportAdapter>>
     */
    protected static array $adapters = [
        'supportpal' => SupportPalAdapter::class,
        'whmcs' => WhmcsAdapter::class,
        'blesta' => BlestaAdapter::class,
        'clientexec' => ClientExecAdapter::class,
        'osticket' => OsTicketAdapter::class,
    ];

    /**
     * Register an adapter class at runtime.
     */
    public static function register(string $class): void
    {
        if (!is_subclass_of($class, DatabaseImportAdapter::class)) {
            throw new \InvalidArgumentException("{$class} must implement DatabaseImportAdapter.");
        }

        $id = $class::id();
        static::$adapters[$id] = $class;
    }

    /**
     * All registered adapters keyed by id.
     *
     * @return array<string,class-string<DatabaseImportAdapter>>
     */
    public static function all(): array
    {
        $map = [];
        foreach (static::$adapters as $key => $class) {
            if (is_string($key)) {
                $map[$key] = $class;
                continue;
            }
            $map[$class::id()] = $class;
        }

        return $map;
    }

    /**
     * Adapter class by identifier.
     */
    public static function adapterClass(string $id): ?string
    {
        return static::all()[$id] ?? null;
    }

    /**
     * Retrieve adapter class by id.
     */
    public static function resolve(string $id): DatabaseImportAdapter
    {
        $class = static::all()[$id] ?? null;
        if (!$class) {
            throw new \InvalidArgumentException("Unknown database import adapter [{$id}].");
        }

        return app($class);
    }

    /**
     * Helper to list adapter metadata for UI.
     *
     * @return array<int,array{id:string,label:string,preview:bool}>
     */
    public static function options(): array
    {
        return Collection::make(static::all())
            ->map(fn (string $class) => [
                'id' => $class::id(),
                'label' => $class::label(),
                'preview' => $class::isPreview(),
            ])
            ->sortBy('label', SORT_NATURAL)
            ->values()
            ->all();
    }
}
