<?php

namespace App\Extensions\Installed\Importer\Services\ApiAdapters;

class ZendeskAdapter extends AbstractApiImportAdapter
{
    public static function id(): string
    {
        return "zendesk";
    }
    public static function label(): string
    {
        return "Zendesk";
    }

    public function boot(array $apiConfig, array $options = []): void
    {
        parent::boot($apiConfig, $options);

        $sub = trim((string) ($this->api["subdomain"] ?? ""));
        $base = rtrim((string) ($this->api["base_url"] ?? ""), "/");
        if ($base === "" && $sub !== "") {
            $base = "https://{$sub}.zendesk.com";
        }
        if ($base === "") {
            throw new \InvalidArgumentException(
                "Zendesk base_url or subdomain is required",
            );
        }

        $email = (string) ($this->api["username"] ?? "");
        $token =
            (string) ($this->api["token"] ?? ($this->api["api_key"] ?? ""));

        if ($email === "" || $token === "") {
            throw new \InvalidArgumentException(
                "Zendesk username (email) and API token are required",
            );
        }

        $this->api["base_url"] = $base;

        $this->api["auth"] = [
            "username" => $email . "/token",
            "password" => $token,
        ];
    }

    public function stages(): array
    {
        $stages = ["departments", "users", "tickets"];
        if (!empty($this->options["import_responses"])) {
            // add comments later
        }
        return $stages;
    }

    public function count(string $stage): int
    {
        try {
            switch ($stage) {
                case "departments":
                    $resp = $this->http()->get("/api/v2/groups");
                    return $this->clampCount(
                        count($resp->json("groups") ?? []),
                    );
                case "users":
                    $resp = $this->http()->get("/api/v2/users/count");
                    return (int) data_get($resp->json(), "count.value", 0);
                case "tickets":
                    $resp = $this->http()->get("/api/v2/tickets/count");
                    return (int) data_get($resp->json(), "count.value", 0);
                default:
                    return 0;
            }
        } catch (\Throwable $e) {
            \Illuminate\Support\Facades\Log::warning("Zendesk API count failed for stage [{$stage}]: {$e->getMessage()}");
            return 0;
        }
    }

    public function fetch(string $stage, int $offset, int $limit): array
    {
        $page = (int) floor($offset / max(1, $limit)) + 1;

        return match ($stage) {
            "departments" => $this->fetchGroups($page, $limit),
            "users" => $this->fetchUsers($page, $limit),
            "tickets" => $this->fetchTickets($page, $limit),
            default => [],
        };
    }

    public function mapStatus(?int $statusId): string
    {
        return "open";
    }

    public function mapPriority(?int $priorityId): string
    {
        return "none";
    }

    public function isOperatorUser(int $userId): bool
    {
        try {
            $resp = $this->http()->get("/api/v2/users/{$userId}");
            $role = strtolower(
                (string) ($resp->json("user.role") ?? "end-user"),
            );
            return $role !== "end-user";
        } catch (\Throwable $e) {
            return false;
        }
    }

    public function findUser(int $userId): ?array
    {
        try {
            $resp = $this->http()->get("/api/v2/users/{$userId}");
            if ($resp->successful()) {
                return $resp->json("user") ?? null;
            }
        } catch (\Throwable $e) {
        }
        return null;
    }

    protected function fetchGroups(int $page, int $perPage): array
    {
        $resp = $this->http()->get("/api/v2/groups", [
            "page" => $page,
            "per_page" => $perPage,
        ]);
        $groups = $resp->json("groups") ?? [];
        return array_map(function (array $g): array {
            return [
                "id" => $g["id"] ?? null,
                "name" => $g["name"] ?? null,
                "public" => 1,
                "email" => null,
                "priority_enabled" => 1,
            ];
        }, $groups);
    }

    protected function fetchUsers(int $page, int $perPage): array
    {
        $resp = $this->http()->get("/api/v2/users", [
            "page" => $page,
            "per_page" => $perPage,
        ]);
        $users = $resp->json("users") ?? [];
        return array_map(function (array $u): array {
            $email = $this->normalizeEmail($u["email"] ?? null);
            return [
                "id" => $u["id"] ?? null,
                "email" => $email,
                "name" => $this->normalizeName($u["name"] ?? null, $email),
                "company" => $u["organization_id"] ?? null,
                "confirmed" => 1,
                "created_at" => $u["created_at"] ?? null,
                "updated_at" => $u["updated_at"] ?? null,
            ];
        }, $users);
    }

    protected function fetchTickets(int $page, int $perPage): array
    {
        $resp = $this->http()->get("/api/v2/tickets", [
            "page" => $page,
            "per_page" => $perPage,
        ]);
        $tickets = $resp->json("tickets") ?? [];

        return array_map(function (array $t): array {
            $priority = strtolower((string) ($t["priority"] ?? ""));
            $priorityId = match ($priority) {
                "low" => 1,
                "normal" => 2,
                "high" => 3,
                "urgent" => 4,
                default => 0,
            };

            $status = strtolower((string) ($t["status"] ?? "open"));
            $statusId = match ($status) {
                "closed" => 5,
                "solved" => 4,
                "pending" => 3,
                "hold" => 4,
                "open" => 2,
                "new" => 1,
                default => 2,
            };

            return [
                "id" => $t["id"] ?? null,
                "number" => (string) ($t["id"] ?? ""),
                "user_id" => $t["requester_id"] ?? null,
                "department_id" => $t["group_id"] ?? null,
                "subject" => $t["subject"] ?? null,
                "message" => $t["description"] ?? "",
                "status_id" => $statusId,
                "priority_id" => $priorityId,
                "cc_list" => $t["email_ccs"] ?? [],
                "assigned_to" => $t["assignee_id"] ?? null,
                "created_at" => $t["created_at"] ?? null,
                "updated_at" => $t["updated_at"] ?? null,
                "last_reply_at" => $t["updated_at"] ?? null,
                "first_staff_reply_at" => null,
                "resolved_at" => null,
                "closed_at" => null,
            ];
        }, $tickets);
    }
}
