<?php

namespace App\Extensions\Installed\Importer\Services\ApiAdapters;

class FreshdeskAdapter extends AbstractApiImportAdapter
{
    public static function id(): string
    {
        return "freshdesk";
    }
    public static function label(): string
    {
        return "Freshdesk";
    }

    public function boot(array $apiConfig, array $options = []): void
    {
        parent::boot($apiConfig, $options);

        $base = rtrim((string) ($this->api["base_url"] ?? ""), "/");
        if ($base === "") {
            throw new \InvalidArgumentException(
                "Freshdesk base_url is required (e.g., https://yourdomain.freshdesk.com)",
            );
        }
        $key = (string) ($this->api["api_key"] ?? "");
        if ($key === "") {
            throw new \InvalidArgumentException(
                "Freshdesk api_key is required",
            );
        }

        $this->api["auth"] = [
            "username" => $key,
            "password" => "X",
        ];
    }

    public function stages(): array
    {
        $stages = ["departments", "users", "tickets"];
        if (!empty($this->options["import_responses"])) {
            // add conversations later
        }
        return $stages;
    }

    public function count(string $stage): int
    {
        try {
            switch ($stage) {
                case "departments":
                    $resp = $this->http()->get("/api/v2/departments");
                    return $this->clampCount(count($resp->json() ?? []));
                case "users":
                    $resp = $this->http()->get("/api/v2/search/contacts", [
                        "query" => '"state:all"',
                    ]);
                    return (int) data_get($resp->json(), "total", 0);
                case "tickets":
                    $resp = $this->http()->get("/api/v2/search/tickets", [
                        "query" => '"*"',
                    ]);
                    return (int) data_get($resp->json(), "total", 0);
                default:
                    return 0;
            }
        } catch (\Throwable $e) {
            \Illuminate\Support\Facades\Log::warning("Freshdesk API count failed for stage [{$stage}]: {$e->getMessage()}");
            return 0;
        }
    }

    public function fetch(string $stage, int $offset, int $limit): array
    {
        $page = (int) floor($offset / max(1, $limit)) + 1;

        return match ($stage) {
            "departments" => $this->fetchDepartments($page, $limit),
            "users" => $this->fetchContacts($page, $limit),
            "tickets" => $this->fetchTickets($page, $limit),
            default => [],
        };
    }

    public function mapStatus(?int $statusId): string
    {
        return match ((int) ($statusId ?? 0)) {
            5 => "closed",
            4 => "in progress",
            3 => "awaiting reply",
            default => "open",
        };
    }

    public function mapPriority(?int $priorityId): string
    {
        return match ((int) ($priorityId ?? 0)) {
            4 => "emergency",
            3 => "high",
            2 => "medium",
            1 => "low",
            default => "none",
        };
    }

    public function isOperatorUser(int $userId): bool
    {
        return false;
    }

    public function findUser(int $userId): ?array
    {
        try {
            $resp = $this->http()->get("/api/v2/contacts/{$userId}");
            if ($resp->successful()) {
                return $resp->json();
            }
        } catch (\Throwable $e) {
        }
        return null;
    }

    protected function fetchDepartments(int $page, int $perPage): array
    {
        $resp = $this->http()->get("/api/v2/departments", [
            "per_page" => $perPage,
            "page" => $page,
        ]);
        $items = $resp->json() ?? [];
        return array_map(function (array $d): array {
            return [
                "id" => $d["id"] ?? null,
                "name" => $d["name"] ?? null,
                "public" => (bool) ($d["is_public"] ?? true),
                "email" => $d["email"] ?? null,
                "priority_enabled" => 1,
            ];
        }, $items);
    }

    protected function fetchContacts(int $page, int $perPage): array
    {
        $resp = $this->http()->get("/api/v2/contacts", [
            "per_page" => $perPage,
            "page" => $page,
        ]);
        $items = $resp->json() ?? [];
        return array_map(function (array $c): array {
            $email = $this->normalizeEmail($c["email"] ?? null);
            return [
                "id" => $c["id"] ?? null,
                "email" => $email,
                "name" => $this->normalizeName($c["name"] ?? null, $email),
                "company" => data_get($c, "company.name"),
                "confirmed" => (int) data_get($c, "active", true) ? 1 : 0,
                "created_at" => $c["created_at"] ?? null,
                "updated_at" => $c["updated_at"] ?? null,
            ];
        }, $items);
    }

    protected function fetchTickets(int $page, int $perPage): array
    {
        $resp = $this->http()->get("/api/v2/tickets", [
            "per_page" => $perPage,
            "page" => $page,
        ]);
        $items = $resp->json() ?? [];
        return array_map(function (array $t): array {
            return [
                "id" => $t["id"] ?? null,
                "number" => (string) ($t["id"] ?? ""),
                "user_id" => $t["requester_id"] ?? null,
                "department_id" => $t["department_id"] ?? null,
                "subject" => $t["subject"] ?? null,
                "message" =>
                    $t["description_text"] ?? ($t["description"] ?? ""),
                "status_id" => $t["status"] ?? null,
                "priority_id" => $t["priority"] ?? null,
                "cc_list" => $t["cc_emails"] ?? [],
                "assigned_to" => $t["responder_id"] ?? null,
                "created_at" => $t["created_at"] ?? null,
                "updated_at" => $t["updated_at"] ?? null,
                "last_reply_at" => $t["updated_at"] ?? null,
                "first_staff_reply_at" => null,
                "resolved_at" => null,
                "closed_at" => null,
            ];
        }, $items);
    }
}
