<?php

namespace App\Extensions\Installed\Importer\Services\ApiAdapters;

use Illuminate\Support\Facades\Http;

abstract class AbstractApiImportAdapter implements ApiImportAdapter
{
    protected array $api = [];
    protected array $options = [];

    protected function http()
    {
        $client = Http::baseUrl(rtrim((string) ($this->api['base_url'] ?? ''), '/'))
            ->acceptJson()
            ->timeout(30)
            ->retry(3, 1000); // Retry 3 times with 1 second delay

        if (!empty($this->api['headers']) && is_array($this->api['headers'])) {
            $client = $client->withHeaders($this->api['headers']);
        }

        if (!empty($this->api['auth']) && is_array($this->api['auth'])) {
            $client = $client->withBasicAuth(
                $this->api['auth']['username'] ?? '',
                $this->api['auth']['password'] ?? ''
            );
        }

        return $client;
    }

    public function boot(array $apiConfig, array $options = []): void
    {
        $this->api = $apiConfig;
        $this->options = $options;
    }

    public static function defaults(): array
    {
        return [
            'chunk_size' => 100,
            'options' => [
                'import_departments' => true,
                'import_categories' => false,
                'import_users' => true,
                'import_operators' => true,
                'import_tickets' => true,
                'import_responses' => false,
                'import_articles' => false,
            ],
        ];
    }

    public static function isPreview(): bool
    {
        return false;
    }

    protected function normalizeEmail(?string $email): ?string
    {
        $email = trim((string) $email);
        return $email !== '' ? strtolower($email) : null;
    }

    protected function normalizeName(?string $name, ?string $emailFallback = null): string
    {
        $name = trim((string) $name);
        if ($name !== '') return $name;
        if ($emailFallback) return ucfirst(strtok($emailFallback, '@')) ?: 'Customer';
        return 'Customer';
    }

    protected function clampCount(int $value): int
    {
        return max(0, $value);
    }
}
