<?php

namespace App\Extensions\Installed\Importer\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

/**
 * Import Mapping Model
 *
 * Stores relationships between SupportPal source IDs and Ticaga records
 * to support resume/retry behaviour during large imports.
 */
class ImportMapping extends Model
{
    protected $table = 'importer_mappings';

    protected $fillable = [
        'job_id',
        'entity_type',
        'source_id',
        'target_id',
        'meta',
    ];

    protected $casts = [
        'meta' => 'array',
    ];

    /**
     * Job relationship.
     */
    public function job(): BelongsTo
    {
        return $this->belongsTo(ImportJob::class, 'job_id');
    }

    /**
     * Persist or update a mapping record.
     */
    public static function remember(
        int $jobId,
        string $entityType,
        string $sourceId,
        ?int $targetId,
        array $meta = []
    ): self {
        return static::updateOrCreate(
            [
                'job_id' => $jobId,
                'entity_type' => $entityType,
                'source_id' => (string) $sourceId,
            ],
            [
                'target_id' => $targetId,
                'meta' => empty($meta) ? null : $meta,
            ]
        );
    }

    /**
     * Resolve a mapped Ticaga ID for a SupportPal entity.
     */
    public static function resolveId(int $jobId, string $entityType, $sourceId): ?int
    {
        return static::where('job_id', $jobId)
            ->where('entity_type', $entityType)
            ->where('source_id', (string) $sourceId)
            ->value('target_id');
    }
}
