<?php

namespace App\Extensions\Installed\Importer\Livewire\Csv;

use Livewire\Component;
use App\Extensions\Installed\Importer\Models\ImportJob;
use App\Extensions\Installed\Importer\Services\CsvImporter;

class Mapping extends Component
{
    public $jobId;
    public $job;
    public $headers = [];
    public $mapping = [];
    public $preview = [];
    public $availableFields = [];

    public function mount($jobId)
    {
        $this->jobId = $jobId;
        $this->job = ImportJob::findOrFail($jobId);

        // Get CSV headers
        $this->headers = $this->job->options['headers'] ?? [];

        // Load existing mapping or create default
        if ($this->job->field_mapping) {
            $this->mapping = $this->job->field_mapping;
        } else {
            // Initialize empty mapping
            foreach ($this->headers as $header) {
                $this->mapping[$header] = $this->guessField($header);
            }
        }

        // Set available fields based on import type
        $this->setAvailableFields();

        // Load preview data
        $this->loadPreview();
    }

    protected function setAvailableFields()
    {
        $fields = [
            'customers' => [
                '' => '-- Skip this column --',
                'name' => 'Name *',
                'email' => 'Email *',
                'company' => 'Company',
                'phone' => 'Phone',
                'phone_number' => 'Phone Number',
                'password' => 'Temporary Password',
                'billing_id' => 'Billing ID',
                'billing_system' => 'Billing System',
                'account_manager' => 'Account Manager ID',
                'force_password_reset' => 'Force Password Reset (true/false)',
            ],
            'departments' => [
                '' => '-- Skip this column --',
                'department_name' => 'Department Name *',
                'department_description' => 'Description',
                'slug' => 'Slug',
                'department_email' => 'Department Email',
                'allows_high_priority' => 'Allows High Priority (true/false)',
                'cc_enabled' => 'CC Enabled (true/false)',
                'is_public' => 'Is Public (true/false)',
                'is_disabled' => 'Is Disabled (true/false)',
                'soft_deleted' => 'Soft Deleted (true/false)',
                'created_at' => 'Created Date',
                'updated_at' => 'Updated Date',
            ],
            'tickets' => [
                '' => '-- Skip this column --',
                'user_id' => 'Customer ID (alternative to email/public email) *',
                'user_email' => 'Customer Email (alternative) *',
                'public_email' => 'Public Email (alternative) *',
                'public_name' => 'Public Name',
                'subject' => 'Subject *',
                'message' => 'Message',
                'department_id' => 'Department ID *',
                'department_name' => 'Department Name (alternative) *',
                'assigned' => 'Assigned Employee ID',
                'status' => 'Status (supports numeric and enum values)',
                'priority' => 'Priority (supports numeric and enum values)',
                'ip_address' => 'IP Address',
                'organize' => 'Source/Organize',
                'created_at' => 'Created Date',
                'updated_at' => 'Updated Date',
            ],
            'responses' => [
                '' => '-- Skip this column --',
                'ticket_id' => 'Ticket ID *',
                'content' => 'Content *',
                'ticket_number' => 'Ticket Number',
                'user_id' => 'User ID',
                'employee_response' => 'Employee Response (true/false)',
                'is_employee' => 'Is Employee Response (true/false)',
                'is_note' => 'Is Internal Note (true/false)',
                'organize' => 'Organize (Type)',
                'ip_address' => 'IP Address',
                'created_at' => 'Created Date',
                'updated_at' => 'Updated Date',
            ],
        ];

        $this->availableFields = $fields[$this->job->type] ?? [];
    }

    protected function guessField($header)
    {
        $header = strtolower(trim($header));
        
        // Common mappings
        $guesses = [
            'name' => 'name',
            'email' => 'email',
            'e-mail' => 'email',
            'company' => 'company',
            'subject' => 'subject',
            'title' => 'title',
            'message' => 'message',
            'body' => 'body',
            'content' => 'content',
            'description' => 'message',
            'customer' => 'user_email',
            'user' => 'user_email',
            'department' => 'department_name',
            'department id' => 'department_id',
            'department_id' => 'department_id',
            'dept_id' => 'department_id',
            'dept id' => 'department_id',
            'status' => 'status',
            'priority' => 'priority',
            'assigned' => 'assigned',
            'assignee' => 'assigned',
            'agent' => 'assigned',
            'ip' => 'ip_address',
            'ip address' => 'ip_address',
            'ticket' => 'ticket_id',
            'ticket_id' => 'ticket_id',
            'ticket number' => 'ticket_number',
            'ticket_number' => 'ticket_number',
            'created' => 'created_at',
            'updated' => 'updated_at',
            'billing' => 'billing_system',
            'billingid' => 'billing_id',
            'accountmanager' => 'account_manager',
            'manager' => 'account_manager',
            'forcepassword' => 'force_password_reset',
            'passwordreset' => 'force_password_reset',
            'temporarypassword' => 'password',
            'temp_password' => 'password',
            'phone' => 'phone_number',
            'public email' => 'public_email',
            'public_email' => 'public_email',
            'public name' => 'public_name',
            'public_name' => 'public_name',
            'organize' => 'organize',
            'source' => 'organize',
            'employee response' => 'employee_response',
            'employeeresponse' => 'employee_response',
            'ip address' => 'ip_address',
        ];

        foreach ($guesses as $pattern => $field) {
            if (str_contains($header, $pattern)) {
                return $field;
            }
        }

        return '';
    }

    protected function loadPreview()
    {
        try {
            $importer = new CsvImporter($this->job->file_path, $this->job->type);
            $this->preview = $importer->preview(5);
        } catch (\Exception $e) {
            $this->preview = [];
        }
    }

    public function saveMapping()
    {
        // Filter out unmapped fields
        $mapping = array_filter($this->mapping, fn($value) => $value !== '');

        // Validate required fields are mapped
        $requiredFields = $this->getRequiredFields();
        $mappedFields = array_values($mapping);
        $missingFields = array_diff($requiredFields, $mappedFields);

        if ($this->job->type === 'responses') {
            if (!in_array('ticket_id', $mappedFields, true) && in_array('ticket_number', $mappedFields, true)) {
                $missingFields = array_diff($missingFields, ['ticket_id']);
            }
        }

        if (!empty($missingFields)) {
            session()->flash('error', 'Please map all required fields: ' . implode(', ', $missingFields));
            return;
        }

        if ($this->job->type === 'tickets') {
            if (
                !in_array('department_id', $mappedFields, true) &&
                !in_array('department_name', $mappedFields, true)
            ) {
                session()->flash('error', 'Please map either Department ID or Department Name.');
                return;
            }

            if (
                !in_array('user_id', $mappedFields, true) &&
                !in_array('user_email', $mappedFields, true) &&
                !in_array('public_email', $mappedFields, true)
            ) {
                session()->flash('error', 'Please map at least one of Customer ID, Customer Email, or Public Email.');
                return;
            }
        } elseif ($this->job->type === 'responses') {
            if (
                !in_array('ticket_id', $mappedFields, true) &&
                !in_array('ticket_number', $mappedFields, true)
            ) {
                session()->flash('error', 'Please map either Ticket ID or Ticket Number.');
                return;
            }
        }

        // Save mapping to job
        $this->job->update([
            'field_mapping' => $mapping,
        ]);

        session()->flash('success', 'Field mapping saved successfully!');

        return redirect()->route('extensions.importer.csv.process', ['jobId' => $this->job->id]);
    }

    protected function getRequiredFields()
    {
        $required = [
            'customers' => ['name', 'email'],
            'departments' => ['department_name'],
            'tickets' => ['subject'],
            'responses' => ['ticket_id', 'content'],
        ];

        return $required[$this->job->type] ?? [];
    }

    public function render()
    {
        return view('extension.importer::csv.mapping')
            ->layout('components.layouts.app');
    }
}
