<?php

namespace App\Extensions\Hooks;

class HookManager
{
    protected array $hooks = [];

    /**
     * Register a hook callback
     */
    public function register(string $hookName, callable $callback, int $priority = 10): void
    {
        if (!isset($this->hooks[$hookName])) {
            $this->hooks[$hookName] = [];
        }

        $this->hooks[$hookName][] = [
            'callback' => $callback,
            'priority' => $priority,
        ];

        // Sort by priority (lower numbers run first)
        usort($this->hooks[$hookName], function ($a, $b) {
            return $a['priority'] <=> $b['priority'];
        });
    }

    /**
     * Execute all callbacks for a specific hook
     */
    public function execute(string $hookName, ...$args): void
    {
        if (!isset($this->hooks[$hookName])) {
            return;
        }

        foreach ($this->hooks[$hookName] as $hook) {
            call_user_func_array($hook['callback'], $args);
        }
    }

    /**
     * Execute all callbacks for a hook and allow them to filter a value
     */
    public function filter(string $hookName, $value, ...$args)
    {
        if (!isset($this->hooks[$hookName])) {
            return $value;
        }

        foreach ($this->hooks[$hookName] as $hook) {
            $value = call_user_func_array($hook['callback'], array_merge([$value], $args));
        }

        return $value;
    }

    /**
     * Remove all callbacks for a specific hook
     */
    public function clear(string $hookName): void
    {
        unset($this->hooks[$hookName]);
    }

    /**
     * Get all registered hooks
     */
    public function all(): array
    {
        return array_keys($this->hooks);
    }

    /**
     * Check if a hook has any registered callbacks
     */
    public function has(string $hookName): bool
    {
        return isset($this->hooks[$hookName]) && count($this->hooks[$hookName]) > 0;
    }
}
