<?php

namespace App\Extensions;

use App\Extensions\Contracts\ExtensionInterface;
use App\Models\Extension as ExtensionModel;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Validation\ValidationException;

class ExtensionManager
{
    protected Collection $extensions;
    protected string $extensionsPath;

    public function __construct()
    {
        $this->extensions = new Collection();
        $this->extensionsPath = app_path('Extensions/Installed');
    }

    /**
     * Discover all available extensions
     */
    public function discover(): self
    {
        if (!File::isDirectory($this->extensionsPath)) {
            File::makeDirectory($this->extensionsPath, 0755, true);
        }

        $directories = File::directories($this->extensionsPath);

        foreach ($directories as $directory) {
            try {
                $this->loadExtension($directory);
            } catch (\Exception $e) {
                Log::error("Failed to load extension from {$directory}: {$e->getMessage()}");
            }
        }

        return $this;
    }

    /**
     * Load a single extension from a directory
     */
    protected function loadExtension(string $path): ?ExtensionInterface
    {
        $manifestPath = $path . '/extension.json';

        if (!File::exists($manifestPath)) {
            Log::warning("No extension.json found in {$path}");
            return null;
        }

        $manifest = json_decode(File::get($manifestPath), true);

        if (!$manifest) {
            Log::warning("Invalid extension.json in {$path}");
            return null;
        }

        // Validate manifest
        try {
            ManifestValidator::validate($manifest);
        } catch (ValidationException $e) {
            Log::error("Invalid manifest in {$path}: " . json_encode($e->errors()));
            return null;
        }

        // Look for the extension class
        $extensionId = $manifest['id'];
        $extensionClass = $this->findExtensionClass($path, $extensionId);

        if (!$extensionClass) {
            Log::warning("Extension class not found for {$extensionId}");
            return null;
        }

        $extension = new $extensionClass($path);

        $this->extensions->put($extensionId, $extension);

        return $extension;
    }

    /**
     * Find the extension class file
     */
    protected function findExtensionClass(string $path, string $extensionId): ?string
    {
        // Convert slug to StudlyCase (e.g., 'my-extension' -> 'MyExtension')
        $className = str($extensionId)->studly() . 'Extension';
        $classFile = $path . '/' . $className . '.php';

        if (!File::exists($classFile)) {
            return null;
        }

        // Build the fully qualified class name
        $namespace = 'App\\Extensions\\Installed\\' . str($extensionId)->studly();
        $fullClassName = $namespace . '\\' . $className;

        // Include the file
        require_once $classFile;

        if (!class_exists($fullClassName)) {
            return null;
        }

        return $fullClassName;
    }

    /**
     * Get all discovered extensions
     */
    public function all(): Collection
    {
        return $this->extensions;
    }

    /**
     * Get a specific extension by ID
     */
    public function get(string $id): ?ExtensionInterface
    {
        return $this->extensions->get($id);
    }

    /**
     * Get all enabled extensions
     */
    public function enabled(): Collection
    {
        return $this->extensions->filter(function (ExtensionInterface $extension) {
            return $this->isEnabled($extension->getId());
        });
    }

    /**
     * Check if an extension is enabled
     */
    public function isEnabled(string $id): bool
    {
        if (!Schema::hasTable('extensions')) {
            return false;
        }

        $record = ExtensionModel::where('extension_id', $id)->first();
        return $record && ($record->installed ?? false) && $record->enabled;
    }

    /**
     * Check if an extension is installed
     */
    public function isInstalled(string $id): bool
    {
        if (!Schema::hasTable('extensions')) {
            return false;
        }

        $record = ExtensionModel::where('extension_id', $id)->first();
        return $record && ($record->installed ?? false);
    }

    /**
     * Install an extension
     */
    public function install(string $id): bool
    {
        $extension = $this->get($id);

        if (!$extension) {
            throw new \RuntimeException("Extension {$id} not found");
        }

        if (!$extension->isCompatible()) {
            throw new \RuntimeException("Extension {$id} is not compatible with this version of Ticaga");
        }

        if ($this->isInstalled($id)) {
            throw new \RuntimeException("Extension {$id} is already installed");
        }

        $existing = ExtensionModel::where('extension_id', $extension->getId())->first();

        $payload = [
            'name' => $extension->getName(),
            'version' => $extension->getVersion(),
            'installed' => true,
            'enabled' => $existing->enabled ?? false,
            'settings' => $existing?->settings ?? ($extension->getConfig() ?? []),
        ];

        ExtensionModel::updateOrCreate(
            ['extension_id' => $extension->getId()],
            $payload
        );

        try {
            // Run the extension's install method
            $extension->install();

            Log::info("Extension {$id} installed successfully");

            return true;
        } catch (\Exception $e) {
            ExtensionModel::where('extension_id', $extension->getId())->update(['installed' => false]);
            Log::error("Failed to install extension {$id}: {$e->getMessage()}");
            throw $e;
        }
    }

    /**
     * Enable an extension
     */
    public function enable(string $id): bool
    {
        $extension = $this->get($id);

        if (!$extension) {
            throw new \RuntimeException("Extension {$id} not found");
        }

        if (!$this->isInstalled($id)) {
            throw new \RuntimeException("Extension {$id} must be installed first");
        }

        if ($this->isEnabled($id)) {
            return true; // Already enabled
        }

        try {
            // Run the extension's enable method
            $extension->enable();

            // Update database record
            ExtensionModel::where('extension_id', $id)->update([
                'enabled' => true,
                'installed' => true,
            ]);

            Log::info("Extension {$id} enabled successfully");

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to enable extension {$id}: {$e->getMessage()}");
            throw $e;
        }
    }

    /**
     * Disable an extension
     */
    public function disable(string $id): bool
    {
        $extension = $this->get($id);

        if (!$extension) {
            throw new \RuntimeException("Extension {$id} not found");
        }

        if (!$this->isEnabled($id)) {
            return true; // Already disabled
        }

        try {
            // Run the extension's disable method
            $extension->disable();

            // Update database record
            ExtensionModel::where('extension_id', $id)->update(['enabled' => false]);

            Log::info("Extension {$id} disabled successfully");

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to disable extension {$id}: {$e->getMessage()}");
            throw $e;
        }
    }

    /**
     * Uninstall an extension
     */
    public function uninstall(string $id): bool
    {
        $extension = $this->get($id);

        if (!$extension) {
            throw new \RuntimeException("Extension {$id} not found");
        }

        if (!$this->isInstalled($id)) {
            throw new \RuntimeException("Extension {$id} is not installed");
        }

        // Disable first if enabled
        if ($this->isEnabled($id)) {
            $this->disable($id);
        }

        try {
            // Run the extension's uninstall method
            $extension->uninstall();

            // Mark as uninstalled (and disabled)
            ExtensionModel::where('extension_id', $id)->update([
                'installed' => false,
                'enabled' => false,
            ]);

            Log::info("Extension {$id} uninstalled successfully");

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to uninstall extension {$id}: {$e->getMessage()}");
            throw $e;
        }
    }

    /**
     * Register all enabled extensions
     */
    public function registerEnabled(): void
    {
        $this->enabled()->each(function (ExtensionInterface $extension) {
            try {
                $extension->registerRoutes();
                $extension->registerViews();
                $extension->registerMigrations();
            } catch (\Exception $e) {
                Log::error("Failed to register extension {$extension->getId()}: {$e->getMessage()}");
            }
        });
    }

    /**
     * Register migrations for all installed extensions
     *
     * Only registers migrations for extensions that are installed,
     * preventing uninstalled extensions from having their migrations run.
     */
    public function registerAllMigrations(): void
    {
        foreach ($this->extensions as $extension) {
            // Only register migrations for installed extensions
            if (!$this->isInstalled($extension->getId())) {
                continue;
            }

            try {
                $extension->registerMigrations();
            } catch (\Exception $e) {
                Log::error("Failed to register migrations for extension {$extension->getId()}: {$e->getMessage()}");
            }
        }
    }

    /**
     * Update extension settings
     */
    public function updateSettings(string $id, array $settings): bool
    {
        if (!$this->isInstalled($id)) {
            throw new \RuntimeException("Extension {$id} is not installed");
        }

        ExtensionModel::where('extension_id', $id)->update([
            'settings' => $settings,
        ]);

        return true;
    }

    /**
     * Get extension settings
     */
    public function getSettings(string $id): array
    {
        $record = ExtensionModel::where('extension_id', $id)->first();

        if (!$record) {
            return [];
        }

        return $record->settings ?? [];
    }

    /**
     * Get installed version of an extension
     */
    public function getInstalledVersion(string $id): ?string
    {
        $record = ExtensionModel::where('extension_id', $id)->first();
        return $record ? $record->version : null;
    }

    /**
     * Check if an upgrade is available for an extension
     */
    public function hasUpgrade(string $id): bool
    {
        if (!$this->isInstalled($id)) {
            return false;
        }

        $extension = $this->get($id);
        if (!$extension) {
            return false;
        }

        $installedVersion = $this->getInstalledVersion($id);
        $currentVersion = $extension->getVersion();

        return version_compare($currentVersion, $installedVersion, '>');
    }

    /**
     * Upgrade an extension to the latest version
     */
    public function upgrade(string $id): bool
    {
        $extension = $this->get($id);

        if (!$extension) {
            throw new \RuntimeException("Extension {$id} not found");
        }

        if (!$this->isInstalled($id)) {
            throw new \RuntimeException("Extension {$id} is not installed. Please install it first.");
        }

        if (!$this->hasUpgrade($id)) {
            throw new \RuntimeException("Extension {$id} is already at the latest version");
        }

        $wasEnabled = $this->isEnabled($id);
        $oldVersion = $this->getInstalledVersion($id);
        $newVersion = $extension->getVersion();

        try {
            // Disable extension during upgrade if it's enabled
            if ($wasEnabled) {
                $this->disable($id);
            }

            // Run any upgrade logic (extensions can override this method)
            if (method_exists($extension, 'upgrade')) {
                $extension->upgrade($oldVersion, $newVersion);
            }

            // Run migrations
            $extension->registerMigrations();

            // Update database record with new version
            ExtensionModel::where('extension_id', $id)->update([
                'version' => $newVersion,
                'name' => $extension->getName(),
            ]);

            // Re-enable if it was enabled before
            if ($wasEnabled) {
                $this->enable($id);
            }

            Log::info("Extension {$id} upgraded from {$oldVersion} to {$newVersion}");

            return true;
        } catch (\Exception $e) {
            Log::error("Failed to upgrade extension {$id}: {$e->getMessage()}");

            // Try to re-enable if it was enabled before and the upgrade failed
            if ($wasEnabled) {
                try {
                    $this->enable($id);
                } catch (\Exception $enableException) {
                    Log::error("Failed to re-enable extension {$id} after failed upgrade: {$enableException->getMessage()}");
                }
            }

            throw $e;
        }
    }
}
