<?php

namespace App\Console\Commands;

use App\Services\Installation\InstallationService;
use Illuminate\Console\Command;

class RecreateLockFile extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'ticaga:create-lock
                            {--force : Force creation even if lock file already exists}';

    /**
     * The console command description.
     */
    protected $description = 'Recreate the installation lock file if it was accidentally deleted';

    protected InstallationService $installation;

    public function __construct(InstallationService $installation)
    {
        parent::__construct();
        $this->installation = $installation;
    }

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Ticaga Lock File Recreation Utility');
        $this->newLine();

        // Check if lock file already exists
        if ($this->installation->isInstalled() && !$this->option('force')) {
            $this->info('✓ Lock file already exists at: storage/installed.lock');
            $this->info('  Use --force to recreate it anyway');
            return self::SUCCESS;
        }

        // Verify installation is actually complete
        if (!$this->verifyInstallation()) {
            $this->error('✗ Installation appears incomplete!');
            $this->error('  Cannot create lock file - please run full installation first');
            $this->newLine();
            $this->info('Run: php artisan ticaga:install');
            return self::FAILURE;
        }

        // Create the lock file
        $this->info('Creating installation lock file...');

        try {
            $this->installation->markInstallationComplete();
            $this->newLine();
            $this->info('✓ Lock file created successfully at: storage/installed.lock');
            $this->info('  Ticaga installation is now marked as complete');

            return self::SUCCESS;
        } catch (\Exception $e) {
            $this->error('✗ Failed to create lock file: ' . $e->getMessage());
            return self::FAILURE;
        }
    }

    /**
     * Verify that the installation is actually complete
     */
    protected function verifyInstallation(): bool
    {
        $checks = [
            'Database reachable' => $this->installation->databaseIsReachable(),
            'Migrations table exists' => $this->installation->hasMigrationsTable(),
            'Users table exists' => $this->installation->hasUsersTable(),
            'Users exist' => $this->installation->hasUsers(),
        ];

        $allPassed = true;

        foreach ($checks as $label => $passed) {
            if ($passed) {
                $this->line("  ✓ {$label}");
            } else {
                $this->error("  ✗ {$label}");
                $allPassed = false;
            }
        }

        $this->newLine();

        return $allPassed;
    }
}
