<?php

namespace App\Console\Commands\Extensions;

use App\Extensions\ExtensionManager;
use Illuminate\Console\Command;

class UpgradeCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'extension:upgrade {id? : The extension ID to upgrade} {--all : Upgrade all extensions with available updates}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Upgrade one or all extensions to their latest version';

    protected ExtensionManager $extensionManager;

    /**
     * Execute the console command.
     */
    public function handle(ExtensionManager $extensionManager): int
    {
        $this->extensionManager = $extensionManager;

        // Discover all extensions
        $this->info('Discovering extensions...');
        $this->extensionManager->discover();

        $extensionId = $this->argument('id');
        $upgradeAll = $this->option('all');

        // Validate input
        if (!$extensionId && !$upgradeAll) {
            $this->error('Please specify an extension ID or use --all to upgrade all extensions');
            $this->info('Usage: php artisan extension:upgrade [extension-id] --all');
            return self::FAILURE;
        }

        if ($extensionId && $upgradeAll) {
            $this->error('Cannot specify both extension ID and --all option');
            return self::FAILURE;
        }

        if ($upgradeAll) {
            return $this->upgradeAll();
        }

        return $this->upgradeSingle($extensionId);
    }

    /**
     * Upgrade a single extension
     */
    protected function upgradeSingle(string $extensionId): int
    {
        $extension = $this->extensionManager->get($extensionId);

        if (!$extension) {
            $this->error("Extension '{$extensionId}' not found");
            return self::FAILURE;
        }

        if (!$this->extensionManager->isInstalled($extensionId)) {
            $this->error("Extension '{$extensionId}' is not installed");
            $this->info("Please install it first with: php artisan extension:install {$extensionId}");
            return self::FAILURE;
        }

        if (!$this->extensionManager->hasUpgrade($extensionId)) {
            $installedVersion = $this->extensionManager->getInstalledVersion($extensionId);
            $this->info("Extension '{$extension->getName()}' is already at the latest version ({$installedVersion})");
            return self::SUCCESS;
        }

        $oldVersion = $this->extensionManager->getInstalledVersion($extensionId);
        $newVersion = $extension->getVersion();

        $this->info("Upgrading '{$extension->getName()}' from {$oldVersion} to {$newVersion}...");

        try {
            $this->extensionManager->upgrade($extensionId);
            $this->info("✓ Successfully upgraded '{$extension->getName()}' to {$newVersion}");
            return self::SUCCESS;
        } catch (\Exception $e) {
            $this->error("✗ Failed to upgrade '{$extension->getName()}': " . $e->getMessage());
            return self::FAILURE;
        }
    }

    /**
     * Upgrade all extensions with available updates
     */
    protected function upgradeAll(): int
    {
        $extensions = $this->extensionManager->all();
        $upgradable = $extensions->filter(function ($extension) {
            return $this->extensionManager->hasUpgrade($extension->getId());
        });

        if ($upgradable->isEmpty()) {
            $this->info('No extensions have available updates');
            return self::SUCCESS;
        }

        $this->info("Found {$upgradable->count()} extension(s) with available updates:");
        $this->newLine();

        // Display table of upgradable extensions
        $tableData = [];
        foreach ($upgradable as $extension) {
            $oldVersion = $this->extensionManager->getInstalledVersion($extension->getId());
            $newVersion = $extension->getVersion();

            $tableData[] = [
                $extension->getName(),
                $extension->getId(),
                $oldVersion,
                $newVersion,
            ];
        }

        $this->table(
            ['Name', 'ID', 'Current Version', 'New Version'],
            $tableData
        );

        $this->newLine();

        if (!$this->confirm('Do you want to upgrade all these extensions?', true)) {
            $this->info('Upgrade cancelled');
            return self::SUCCESS;
        }

        $this->newLine();
        $successCount = 0;
        $failureCount = 0;

        foreach ($upgradable as $extension) {
            $extensionId = $extension->getId();
            $oldVersion = $this->extensionManager->getInstalledVersion($extensionId);
            $newVersion = $extension->getVersion();

            $this->info("Upgrading '{$extension->getName()}' from {$oldVersion} to {$newVersion}...");

            try {
                $this->extensionManager->upgrade($extensionId);
                $this->info("✓ Successfully upgraded '{$extension->getName()}'");
                $successCount++;
            } catch (\Exception $e) {
                $this->error("✗ Failed to upgrade '{$extension->getName()}': " . $e->getMessage());
                $failureCount++;
            }

            $this->newLine();
        }

        // Summary
        $this->info("Upgrade complete:");
        $this->info("  - Successful: {$successCount}");
        if ($failureCount > 0) {
            $this->error("  - Failed: {$failureCount}");
        }

        return $failureCount > 0 ? self::FAILURE : self::SUCCESS;
    }
}
