<?php

namespace App\Console\Commands\Extensions;

use App\Models\Extension;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class ListCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'extension:list {--available : Show available extensions not yet installed}';

    /**
     * The console command description.
     */
    protected $description = 'List all extensions';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        if ($this->option('available')) {
            return $this->listAvailableExtensions();
        }

        return $this->listInstalledExtensions();
    }

    /**
     * List installed extensions.
     */
    protected function listInstalledExtensions(): int
    {
        $extensions = Extension::orderBy('extension_id')->get();

        if ($extensions->isEmpty()) {
            $this->info('No extensions installed.');
            $this->newLine();
            $this->info('To see available extensions: php artisan extension:list --available');
            return 0;
        }

        $this->info('Installed Extensions:');
        $this->newLine();

        $rows = [];
        foreach ($extensions as $extension) {
            $rows[] = [
                $extension->extension_id,
                $extension->name,
                $extension->version ?? 'N/A',
                $extension->installed ? '✓' : '✗',
                $extension->enabled ? '✓' : '✗',
            ];
        }

        $this->table(
            ['ID', 'Name', 'Version', 'Installed', 'Enabled'],
            $rows
        );

        return 0;
    }

    /**
     * List available extensions in the filesystem.
     */
    protected function listAvailableExtensions(): int
    {
        $installedPath = app_path('Extensions/Installed');

        if (!File::isDirectory($installedPath)) {
            $this->error('Extensions directory not found.');
            return 1;
        }

        $directories = File::directories($installedPath);
        $available = [];

        foreach ($directories as $dir) {
            $extensionName = basename($dir);
            // Convert StudlyCase to kebab-case
            $extensionId = strtolower(preg_replace('/([a-z])([A-Z])/', '$1-$2', $extensionName));

            $extensionFile = $dir . "/{$extensionName}Extension.php";

            if (File::exists($extensionFile)) {
                $extension = Extension::where('extension_id', $extensionId)->first();
                $installed = $extension && $extension->installed;
                $enabled = $extension && $extension->enabled;

                $available[] = [
                    'id' => $extensionId,
                    'name' => $extensionName,
                    'path' => $dir,
                    'installed' => $installed,
                    'enabled' => $enabled,
                ];
            }
        }

        if (empty($available)) {
            $this->info('No extensions found in app/Extensions/Installed/');
            return 0;
        }

        $this->info('Available Extensions:');
        $this->newLine();

        $rows = [];
        foreach ($available as $ext) {
            $status = 'Not Installed';
            $action = "php artisan extension:install {$ext['id']}";

            if ($ext['installed'] && $ext['enabled']) {
                $status = '✓ Installed & Enabled';
                $action = '';
            } elseif ($ext['installed']) {
                $status = '○ Installed Only';
                $action = "php artisan extension:enable {$ext['id']}";
            }

            $rows[] = [
                $ext['id'],
                $ext['name'],
                $status,
                $action,
            ];
        }

        $this->table(
            ['ID', 'Name', 'Status', 'Next Action'],
            $rows
        );

        return 0;
    }
}
