<?php

namespace App\Console\Commands\Extensions;

use App\Models\Extension;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class InstallCommand extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'extension:install {id : The extension ID to install} {--force : Force reinstallation}';

    /**
     * The console command description.
     */
    protected $description = 'Install an extension';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $extensionId = $this->argument('id');
        $force = $this->option('force');

        $this->info("Installing extension: {$extensionId}");

        try {
            // Find or create extension record
            $extension = Extension::where('extension_id', $extensionId)->first();

            if (!$extension) {
                // Try to load extension class to get metadata
                $extensionClass = $this->findExtensionClass($extensionId);
                if (!$extensionClass) {
                    $this->error("Extension '{$extensionId}' not found!");
                    $this->line('Run: php artisan extension:list to see available extensions');
                    return 1;
                }

                $this->info('Creating extension record...');
                $extension = Extension::create([
                    'extension_id' => $extensionId,
                    'name' => $this->getExtensionName($extensionId),
                    'version' => '1.0.0',
                    'installed' => false,
                    'enabled' => false,
                ]);
            }

            // Check if already installed
            if ($extension->installed && !$force) {
                $this->warn("Extension '{$extensionId}' is already installed!");
                $this->info('Use --force to reinstall');
                return 1;
            }

            // Load extension class
            $extensionClass = $this->findExtensionClass($extensionId);
            if (!$extensionClass) {
                $this->error("Extension class not found for '{$extensionId}'");
                return 1;
            }

            // Get extension path
            $extensionPath = $this->getExtensionPath($extensionId);

            // Instantiate extension with path
            $instance = new $extensionClass($extensionPath);

            // Set extension context using reflection
            $reflection = new \ReflectionClass($instance);
            $property = $reflection->getProperty('extension');
            $property->setAccessible(true);
            $property->setValue($instance, $extension);

            // Check requirements
            $this->info('Checking requirements...');
            $requirementCheck = $instance->checkRequirements();

            // Display requirements
            if (!empty($requirementCheck['errors'])) {
                $this->error('❌ Requirements not met:');
                foreach ($requirementCheck['errors'] as $error) {
                    $this->line("  • {$error}");
                }
                $this->newLine();
            }

            if (!empty($requirementCheck['warnings'])) {
                $this->warn('⚠️  Warnings:');
                foreach ($requirementCheck['warnings'] as $warning) {
                    $this->line("  • {$warning}");
                }
                $this->newLine();
            }

            if ($requirementCheck['passed']) {
                $this->info('✓ All requirements met');
                $this->newLine();
            } else {
                if (!$force && !$this->confirm('Requirements not met. Continue anyway?', false)) {
                    $this->warn('Installation cancelled.');
                    return 1;
                }
                $this->warn('⚠️  Continuing with unmet requirements...');
                $this->newLine();
            }

            // Run install method
            $this->info('Running installation process...');
            $instance->install();

            // Mark as installed
            $extension->update(['installed' => true]);

            $this->newLine();
            $this->info("✓ Extension '{$extensionId}' installed successfully!");
            $this->newLine();
            $this->info('Next steps:');
            $this->line("  php artisan extension:enable {$extensionId}");

            return 0;

        } catch (\Exception $e) {
            $this->error('Installation failed: ' . $e->getMessage());
            $this->newLine();
            $this->error('Stack trace:');
            $this->line($e->getTraceAsString());
            return 1;
        }
    }

    /**
     * Find the extension class file.
     */
    protected function findExtensionClass(string $extensionId): ?string
    {
        // Convert kebab-case to StudlyCase (e.g., live-chat -> LiveChat)
        $className = str_replace('-', '', ucwords($extensionId, '-'));

        $path = app_path("Extensions/Installed/{$className}/{$className}Extension.php");

        if (!File::exists($path)) {
            return null;
        }

        $class = "App\\Extensions\\Installed\\{$className}\\{$className}Extension";

        if (!class_exists($class)) {
            return null;
        }

        return $class;
    }

    /**
     * Get the extension directory path.
     */
    protected function getExtensionPath(string $extensionId): string
    {
        // Convert kebab-case to StudlyCase (e.g., live-chat -> LiveChat)
        $className = str_replace('-', '', ucwords($extensionId, '-'));
        return app_path("Extensions/Installed/{$className}");
    }

    /**
     * Get a human-readable extension name from ID.
     */
    protected function getExtensionName(string $extensionId): string
    {
        return ucwords(str_replace('-', ' ', $extensionId));
    }
}
